/*
 * Decompiled with CFR 0.152.
 */
package com.bankmemory.data;

import com.bankmemory.data.BankItem;
import com.bankmemory.data.BankSave;
import com.bankmemory.data.BankWorldType;
import com.bankmemory.data.DataStoreUpdateListener;
import com.bankmemory.data.DisplayNameMapper;
import com.bankmemory.data.ItemDataParser;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;
import com.google.gson.reflect.TypeToken;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.runelite.client.config.ConfigManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class PluginDataStore {
    private static final Logger log = LoggerFactory.getLogger(PluginDataStore.class);
    private static final String PLUGIN_BASE_GROUP = "bankMemory";
    private static final String CURRENT_LIST_KEY = "currentList";
    private static final String SNAPSHOT_LIST_KEY = "snapshotList";
    private static final String NAME_MAP_KEY = "nameMap";
    private final Object dataLock = new Object();
    private final ConfigManager configManager;
    private final ItemDataParser itemDataParser;
    private final Map<String, String> nameMap = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
    private final List<BankSave> currentBankList;
    private final List<BankSave> snapshotBanksList;
    private final BlockingQueue<ConfigWrite> configWritesQueue = new LinkedBlockingQueue<ConfigWrite>();
    private final List<DataStoreUpdateListener> listeners = new ArrayList<DataStoreUpdateListener>();

    @Inject
    private PluginDataStore(ConfigManager configManager, ItemDataParser itemDataParser) {
        this.configManager = configManager;
        this.itemDataParser = itemDataParser;
        this.currentBankList = this.loadCurrentBankList();
        this.snapshotBanksList = this.loadSnapshotBanksList();
        this.nameMap.putAll(this.loadNameMapData());
        Thread configWriter = new Thread((Runnable)new ConfigWriter(), "Bank Memory config writer");
        configWriter.setDaemon(true);
        configWriter.start();
    }

    private List<BankSave> loadCurrentBankList() {
        Type deserialiseType = new TypeToken<List<BankSave>>(){}.getType();
        List fromDataStore = this.loadDataFromConfig(CURRENT_LIST_KEY, deserialiseType, new ArrayList(), "Current bank list");
        return this.upgradeBankSaves(fromDataStore);
    }

    private <T> T loadDataFromConfig(String configKey, Type deserialiseType, T defaultInstance, String dataName) {
        String jsonString = this.configManager.getConfiguration(PLUGIN_BASE_GROUP, configKey);
        if (jsonString == null) {
            return defaultInstance;
        }
        Gson gson = this.buildGson();
        try {
            Object loadedData = gson.fromJson(jsonString, deserialiseType);
            return (T)(loadedData == null ? defaultInstance : loadedData);
        }
        catch (JsonParseException ex) {
            log.error("{} json invalid. All is lost", (Object)dataName, (Object)ex);
            this.configManager.unsetConfiguration(PLUGIN_BASE_GROUP, configKey);
            return defaultInstance;
        }
    }

    private List<BankSave> upgradeBankSaves(List<BankSave> bankSaves) {
        return bankSaves.stream().map(BankSave::cleanItemData).collect(Collectors.toList());
    }

    private List<BankSave> loadSnapshotBanksList() {
        Type deserialiseType = new TypeToken<List<BankSave>>(){}.getType();
        List fromDataStore = this.loadDataFromConfig(SNAPSHOT_LIST_KEY, deserialiseType, new ArrayList(), "Snapshot bank list");
        return this.upgradeBankSaves(fromDataStore);
    }

    private Gson buildGson() {
        Type itemDataListType = new TypeToken<ImmutableList<BankItem>>(){}.getType();
        return new GsonBuilder().registerTypeAdapter(itemDataListType, (Object)this.itemDataParser).create();
    }

    private Map<String, String> loadNameMapData() {
        Type deserialiseType = new TypeToken<HashMap<String, String>>(){}.getType();
        return this.loadDataFromConfig(NAME_MAP_KEY, deserialiseType, new HashMap(), "Display name map");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void registerDisplayNameForLogin(String login, String displayName) {
        boolean changed;
        ArrayList<DataStoreUpdateListener> listenersCopy;
        Object object = this.dataLock;
        synchronized (object) {
            listenersCopy = new ArrayList<DataStoreUpdateListener>(this.listeners);
            String oldValue = this.nameMap.put(login, displayName);
            boolean bl = changed = !Objects.equals(oldValue, displayName);
            if (changed) {
                ConfigWrite write = new ConfigWrite(PLUGIN_BASE_GROUP, NAME_MAP_KEY, new HashMap<String, String>(this.nameMap));
                this.scheduleConfigWrite(write);
            }
        }
        if (changed) {
            listenersCopy.forEach(DataStoreUpdateListener::displayNameMapUpdated);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DisplayNameMapper getDisplayNameMapper() {
        Object object = this.dataLock;
        synchronized (object) {
            return new DisplayNameMapper(this.nameMap);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addListener(DataStoreUpdateListener listener) {
        Object object = this.dataLock;
        synchronized (object) {
            this.listeners.add(listener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeListener(DataStoreUpdateListener listener) {
        Object object = this.dataLock;
        synchronized (object) {
            this.listeners.remove(listener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Optional<BankSave> getDataForCurrentBank(BankWorldType worldType, String login) {
        if (Strings.isNullOrEmpty((String)login)) {
            return Optional.empty();
        }
        Object object = this.dataLock;
        synchronized (object) {
            return this.currentBankList.stream().filter(s -> s.getWorldType() == worldType && s.getUserName().equalsIgnoreCase(login)).findAny();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<BankSave> getCurrentBanksList() {
        Object object = this.dataLock;
        synchronized (object) {
            return new ArrayList<BankSave>(this.currentBankList);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<BankSave> getSnapshotBanksList() {
        Object object = this.dataLock;
        synchronized (object) {
            return new ArrayList<BankSave>(this.snapshotBanksList);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Optional<BankSave> getBankSaveWithId(long id) {
        Object object = this.dataLock;
        synchronized (object) {
            return Stream.concat(this.currentBankList.stream(), this.snapshotBanksList.stream()).filter(s -> s.getId() == id).findFirst();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void saveAsCurrentBank(BankSave newSave) {
        ArrayList<DataStoreUpdateListener> listenersCopy;
        Object object = this.dataLock;
        synchronized (object) {
            listenersCopy = new ArrayList<DataStoreUpdateListener>(this.listeners);
            this.saveAsCurrentBankImpl(newSave);
        }
        listenersCopy.forEach(DataStoreUpdateListener::currentBanksListChanged);
    }

    private void saveAsCurrentBankImpl(BankSave newSave) {
        this.currentBankList.stream().filter(s -> s.getUserName().equalsIgnoreCase(newSave.getUserName()) && s.getWorldType() == newSave.getWorldType()).findAny().ifPresent(this.currentBankList::remove);
        this.currentBankList.add(0, newSave);
        ConfigWrite configWrite = new ConfigWrite(PLUGIN_BASE_GROUP, CURRENT_LIST_KEY, new ArrayList<BankSave>(this.currentBankList));
        this.scheduleConfigWrite(configWrite);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void saveAsSnapshotBank(String newName, BankSave existingSave) {
        ArrayList<DataStoreUpdateListener> listenersCopy;
        Object object = this.dataLock;
        synchronized (object) {
            listenersCopy = new ArrayList<DataStoreUpdateListener>(this.listeners);
            this.snapshotBanksList.add(0, BankSave.snapshotFromExistingBank(newName, existingSave));
            ConfigWrite configWrite = new ConfigWrite(PLUGIN_BASE_GROUP, SNAPSHOT_LIST_KEY, new ArrayList<BankSave>(this.snapshotBanksList));
            this.scheduleConfigWrite(configWrite);
        }
        listenersCopy.forEach(DataStoreUpdateListener::snapshotBanksListChanged);
    }

    private void scheduleConfigWrite(ConfigWrite configWrite) {
        try {
            log.debug("Scheduling write for {}.{}", (Object)configWrite.configGroup, (Object)configWrite.configKey);
            this.configWritesQueue.put(configWrite);
        }
        catch (InterruptedException ex) {
            log.error("Unexpected interrupt whilst schedule config write. Data not being written", (Throwable)ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteBankSaveWithId(long saveId) {
        boolean changed;
        ArrayList<DataStoreUpdateListener> listenersCopy;
        Object object = this.dataLock;
        synchronized (object) {
            listenersCopy = new ArrayList<DataStoreUpdateListener>(this.listeners);
            changed = this.deleteBankSaveWithIdImpl(saveId, this.currentBankList, CURRENT_LIST_KEY) || this.deleteBankSaveWithIdImpl(saveId, this.snapshotBanksList, SNAPSHOT_LIST_KEY);
        }
        if (changed) {
            listenersCopy.forEach(DataStoreUpdateListener::currentBanksListChanged);
        } else {
            log.error("Tried deleting missing bank save: {}", (Object)saveId);
        }
    }

    private boolean deleteBankSaveWithIdImpl(long id, List<BankSave> saveList, String listConfigKey) {
        Optional<BankSave> save = saveList.stream().filter(s -> s.getId() == id).findFirst();
        if (save.isPresent()) {
            saveList.remove(save.get());
            ConfigWrite configWrite = new ConfigWrite(PLUGIN_BASE_GROUP, listConfigKey, new ArrayList<BankSave>(saveList));
            this.scheduleConfigWrite(configWrite);
            return true;
        }
        return false;
    }

    private class ConfigWriter
    implements Runnable {
        private ConfigWriter() {
        }

        @Override
        public void run() {
            Gson gson = PluginDataStore.this.buildGson();
            while (!Thread.interrupted()) {
                ConfigWrite write;
                try {
                    write = PluginDataStore.this.configWritesQueue.take();
                    log.debug("Got write for {}.{}", (Object)write.configGroup, (Object)write.configKey);
                }
                catch (InterruptedException ex) {
                    log.warn("ConfigWriter thread interrupted", (Throwable)ex);
                    break;
                }
                PluginDataStore.this.configManager.setConfiguration(write.configGroup, write.configKey, gson.toJson(write.data));
            }
        }
    }

    private static class ConfigWrite {
        final String configGroup;
        final String configKey;
        final Object data;

        public ConfigWrite(String configGroup, String configKey, Object data) {
            this.configGroup = configGroup;
            this.configKey = configKey;
            this.data = data;
        }
    }
}

