/*
 * Decompiled with CFR 0.152.
 */
package com.gpperhour;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.gpperhour.GPPerHourConfig;
import com.gpperhour.GPPerHourPlugin;
import com.gpperhour.LedgerItem;
import com.gpperhour.RunState;
import com.gpperhour.SessionStats;
import com.gpperhour.TripData;
import java.lang.reflect.Type;
import java.time.Instant;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionManager {
    private static final Logger log = LoggerFactory.getLogger(SessionManager.class);
    private final GPPerHourPlugin plugin;
    private final GPPerHourConfig config;
    private final ScheduledExecutorService executor;
    private final Gson gson;
    private final Map<String, TripData> activeTrips = new HashMap<String, TripData>();
    private String activeSessionStartId;
    private String activeSessionEndId;
    List<SessionStats> sessionHistory = new LinkedList<SessionStats>();
    List<String> savedSessionIdentifiers = null;
    boolean sessionHistoryDirty;

    public SessionManager(GPPerHourPlugin plugin, GPPerHourConfig config, ScheduledExecutorService executor, Gson gson) {
        this.plugin = plugin;
        this.config = config;
        this.executor = executor;
        this.gson = gson;
    }

    void refreshSessionTracking() {
        if (this.isTracking()) {
            this.startTracking();
        } else {
            this.stopTracking();
        }
    }

    boolean isTracking() {
        return this.config.getEnableSessionTracking() && this.config.enableSessionPanel();
    }

    void startTracking() {
        if (this.plugin.getState() == RunState.BANK) {
            return;
        }
        TripData activeTrip = this.plugin.getRunData();
        if (activeTrip != null && !activeTrip.isBankDelay && !activeTrip.isFirstRun && activeTrip.isInProgress()) {
            this.onTripStarted(activeTrip);
        }
    }

    void stopTracking() {
        String activeTripIdentifier = null;
        for (TripData trip : this.activeTrips.values()) {
            if (!trip.isInProgress()) continue;
            activeTripIdentifier = trip.identifier;
        }
        if (activeTripIdentifier != null) {
            this.deleteTrip(activeTripIdentifier);
        }
    }

    private List<TripData> getSortedTrips() {
        return this.activeTrips.values().stream().sorted(Comparator.comparingLong(o -> o.runStartTime)).collect(Collectors.toList());
    }

    SessionStats getActiveSessionStats() {
        if (this.activeSessionStartId == null) {
            return null;
        }
        List<TripData> runDataSorted = this.getSortedTrips();
        long gains = 0L;
        long losses = 0L;
        long tripDurationSum = 0L;
        long totalPauseTime = 0L;
        boolean foundStart = false;
        int tripCount = 0;
        HashMap<Integer, Float> initialQtys = new HashMap<Integer, Float>();
        HashMap<Integer, Float> qtys = new HashMap<Integer, Float>();
        for (TripData runData : runDataSorted) {
            if (!(foundStart |= runData.identifier.equals(this.activeSessionStartId))) continue;
            for (Integer initialId : runData.initialItemQtys.keySet()) {
                initialQtys.merge(initialId, runData.initialItemQtys.get(initialId), Float::sum);
            }
            for (Integer itemId : runData.itemQtys.keySet()) {
                qtys.merge(itemId, runData.itemQtys.get(itemId), Float::sum);
            }
            List<LedgerItem> ledger = GPPerHourPlugin.getProfitLossLedger(runData.initialItemQtys, runData.itemQtys);
            for (LedgerItem item : ledger) {
                long value = item.getCombinedValue();
                if (value > 0L) {
                    gains += value;
                    continue;
                }
                losses += value;
            }
            tripDurationSum += runData.getRuntime();
            totalPauseTime += runData.getEndTime() - runData.runStartTime - runData.getRuntime();
            ++tripCount;
            if (this.activeSessionEndId == null || !this.activeSessionEndId.equals(runData.identifier)) continue;
            break;
        }
        if (!foundStart) {
            log.error("couldn't find start session");
            return null;
        }
        long sessionRuntime = 0L;
        if (this.config.ignoreBankTime()) {
            sessionRuntime = tripDurationSum;
        } else {
            long sessionStartTime = this.getSessionStartTime();
            long sessionEndTime = this.getSessionEndTime();
            sessionRuntime = sessionEndTime - sessionStartTime - totalPauseTime;
        }
        long netTotal = gains + losses;
        long avgTripDuration = (long)((float)tripDurationSum / (float)tripCount);
        return new SessionStats(this.getSessionEndTime(), sessionRuntime, gains, losses, netTotal, tripCount, avgTripDuration, initialQtys, qtys);
    }

    long getSessionStartTime() {
        if (this.activeSessionStartId == null) {
            return 0L;
        }
        return this.getSessionStartTrip().runStartTime;
    }

    long getSessionEndTime() {
        return this.activeSessionEndId == null ? Instant.now().toEpochMilli() : (this.activeTrips.get(this.activeSessionEndId).isInProgress() ? Instant.now().toEpochMilli() : this.activeTrips.get((Object)this.activeSessionEndId).runEndTime.longValue());
    }

    boolean isTimeInActiveSession(long time) {
        long startTime = this.getSessionStartTime();
        long endTime = this.getSessionEndTime();
        return time >= startTime && time <= endTime;
    }

    void setSessionStart(String id) {
        this.activeSessionStartId = id;
        if (id != null) {
            TripData startTrip = this.getSessionStartTrip();
            TripData endtrip = this.getSessionEndTrip();
            if (endtrip != null && endtrip.runStartTime < startTrip.runStartTime) {
                this.setSessionEnd(id);
            }
        }
    }

    void setSessionEnd(String id) {
        this.activeSessionEndId = id;
        if (id != null) {
            TripData startTrip = this.getSessionStartTrip();
            TripData endtrip = this.getSessionEndTrip();
            if (startTrip != null && startTrip.runStartTime > endtrip.runStartTime) {
                this.setSessionStart(id);
            }
        }
    }

    TripData getSessionStartTrip() {
        if (this.activeSessionStartId == null) {
            return null;
        }
        return this.activeTrips.get(this.activeSessionStartId);
    }

    TripData getSessionEndTrip() {
        if (this.activeSessionEndId == null) {
            if (this.activeTrips.size() == 0) {
                return null;
            }
            List<TripData> sortedData = this.getSortedTrips();
            return sortedData.get(sortedData.size() - 1);
        }
        return this.activeTrips.get(this.activeSessionEndId);
    }

    void deleteAllTrips() {
        LinkedList<TripData> allTrips = new LinkedList<TripData>(this.activeTrips.values());
        this.activeSessionStartId = null;
        this.activeSessionEndId = null;
        for (TripData trip : allTrips) {
            if (trip.isInProgress()) {
                this.activeSessionStartId = trip.identifier;
                continue;
            }
            this.activeTrips.remove(trip.identifier);
        }
    }

    void deleteTrip(String id) {
        this.activeTrips.remove(id);
        if (this.activeSessionStartId == id) {
            this.activeSessionStartId = this.activeTrips.size() == 0 ? null : this.getSortedTrips().get((int)0).identifier;
        }
        if (this.activeSessionEndId == id) {
            this.activeSessionEndId = null;
        }
    }

    void onTripStarted(TripData runData) {
        if (!this.isTracking()) {
            return;
        }
        this.activeTrips.put(runData.identifier, runData);
        if (this.activeSessionStartId == null) {
            this.activeSessionStartId = runData.identifier;
        }
    }

    void onTripCompleted(TripData runData) {
        if (!this.isTracking()) {
            return;
        }
        if (!this.tripHadChange(runData.initialItemQtys, runData.itemQtys)) {
            log.debug("nothing changed, ignoring trip");
            this.deleteTrip(runData.identifier);
            return;
        }
    }

    boolean tripHadChange(Map<Integer, Float> tripStart, Map<Integer, Float> tripEnd) {
        List<LedgerItem> ledger = GPPerHourPlugin.getProfitLossLedger(tripStart, tripEnd);
        return !ledger.isEmpty();
    }

    void saveNewSession(String name) {
        if (this.savedSessionIdentifiers == null) {
            log.error("can't save session, hasn't loaded sessions yet.");
            return;
        }
        this.executor.execute(() -> {
            SessionStats statsToSave = this.getActiveSessionStats();
            if (statsToSave == null) {
                return;
            }
            statsToSave.sessionName = name;
            statsToSave.sessionID = UUID.randomUUID().toString();
            this.sessionHistory.add(statsToSave);
            String json = this.gson.toJson((Object)statsToSave);
            this.plugin.saveData(GPPerHourConfig.getSessionKey(statsToSave.sessionID), json);
            this.savedSessionIdentifiers.add(statsToSave.sessionID);
            this.saveSessionIdentifiers();
            this.sessionHistoryDirty = true;
        });
    }

    void overwriteSession(SessionStats sessionStats) {
        if (sessionStats == null) {
            return;
        }
        this.executor.execute(() -> {
            String json = this.gson.toJson((Object)sessionStats);
            this.plugin.saveData(GPPerHourConfig.getSessionKey(sessionStats.sessionID), json);
            this.sessionHistoryDirty = true;
        });
    }

    void deleteSession(SessionStats sessionStats) {
        if (this.savedSessionIdentifiers == null) {
            log.error("can't delete session, hasn't loaded sessions yet.");
            return;
        }
        if (sessionStats == null) {
            return;
        }
        this.executor.execute(() -> {
            this.sessionHistory.remove(sessionStats);
            this.plugin.deleteData(GPPerHourConfig.getSessionKey(sessionStats.sessionID));
            this.savedSessionIdentifiers.remove(sessionStats.sessionID);
            this.saveSessionIdentifiers();
            this.sessionHistoryDirty = true;
        });
    }

    void saveSessionIdentifiers() {
        String json = this.gson.toJson(this.savedSessionIdentifiers);
        this.plugin.saveData("session_ids", json);
    }

    void reloadSessions() {
        this.sessionHistory.clear();
        this.savedSessionIdentifiers = null;
        this.executor.execute(() -> {
            Type listType = new TypeToken<List<String>>(){}.getType();
            String keysJSON = this.plugin.readData("session_ids");
            try {
                this.savedSessionIdentifiers = (List)this.gson.fromJson(keysJSON, listType);
            }
            catch (Exception e) {
                log.error("Failed to load session identifiers from json: " + keysJSON, (Throwable)e);
                return;
            }
            if (this.savedSessionIdentifiers == null) {
                this.savedSessionIdentifiers = new LinkedList<String>();
            }
            for (String sessionIdentifier : this.savedSessionIdentifiers) {
                String json = this.plugin.readData(GPPerHourConfig.getSessionKey(sessionIdentifier));
                if (json == null) continue;
                try {
                    SessionStats sessionStats = (SessionStats)this.gson.fromJson(json, SessionStats.class);
                    this.sessionHistory.add(sessionStats);
                }
                catch (Exception e) {
                    log.error("Failed to load session data from json: " + json, (Throwable)e);
                    return;
                }
            }
            this.sessionHistoryDirty = true;
        });
    }

    public Map<String, TripData> getActiveTrips() {
        return this.activeTrips;
    }

    public String getActiveSessionStartId() {
        return this.activeSessionStartId;
    }

    public String getActiveSessionEndId() {
        return this.activeSessionEndId;
    }
}

