/*
 * Decompiled with CFR 0.152.
 */
package net.machpi.runelite.influxdb.write;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import java.util.ArrayDeque;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicReference;
import javax.inject.Inject;
import javax.inject.Singleton;
import net.machpi.runelite.influxdb.InfluxDbConfig;
import net.machpi.runelite.influxdb.MeasurementCreator;
import net.machpi.runelite.influxdb.write.Measurement;
import net.machpi.runelite.influxdb.write.Series;
import org.apache.commons.lang3.StringUtils;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBFactory;
import org.influxdb.dto.BatchPoints;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class InfluxWriter {
    private static final Logger log = LoggerFactory.getLogger(InfluxWriter.class);
    private final InfluxDbConfig config;
    private final ConcurrentMap<Series, Writer> writers = new ConcurrentHashMap<Series, Writer>();
    private InfluxDB cachedServer = null;
    private String serverUrl;
    private String serverUser;
    private String serverPass;
    private static final FilterOp FULL_DEDUPE = (prev, b) -> prev == null || !prev.getNumericValues().equals(b.getNumericValues()) || !prev.getStringValues().equals(b.getStringValues());
    private static final FilterOp SELF_DEDUPE = (prev, curr) -> {
        if (prev == null) {
            return true;
        }
        if (!prev.getStringValues().equals(curr.getStringValues())) {
            return true;
        }
        for (String posKey : MeasurementCreator.SELF_POS_KEYS) {
            Number p = prev.getNumericValues().get(posKey);
            Number c = curr.getNumericValues().get(posKey);
            if (p == null || c == null) {
                return true;
            }
            if (!(Math.abs(p.doubleValue() - c.doubleValue()) > 5.0)) continue;
            return true;
        }
        return !Objects.equals(Maps.filterKeys(prev.getNumericValues(), x -> !MeasurementCreator.SELF_POS_KEYS.contains(x)), Maps.filterKeys(curr.getNumericValues(), x -> !MeasurementCreator.SELF_POS_KEYS.contains(x)));
    };

    @Inject
    public InfluxWriter(InfluxDbConfig config) {
        this.config = config;
    }

    public void submit(Measurement m) {
        this.writer(m.getSeries()).submit(m);
    }

    public boolean isBlocked(Series s) {
        return this.writer(s).isBlocked();
    }

    private synchronized Optional<InfluxDB> getInflux() {
        if (StringUtils.isEmpty((CharSequence)this.config.getDatabase())) {
            return Optional.empty();
        }
        String newServerUrl = this.config.getServerUrl();
        String newServerUser = this.config.getServerUsername();
        String newServerPass = this.config.getServerPassword();
        if (!(Objects.equals(newServerUrl, this.serverUrl) && Objects.equals(newServerUser, this.serverUser) && Objects.equals(newServerPass, this.serverPass) && this.cachedServer != null)) {
            if (this.cachedServer != null) {
                this.cachedServer.close();
                this.cachedServer = null;
                this.serverUrl = null;
                this.serverUser = null;
                this.serverPass = null;
            }
            if (!StringUtils.isEmpty((CharSequence)this.serverUrl)) {
                this.cachedServer = StringUtils.isEmpty((CharSequence)this.serverPass) || StringUtils.isEmpty((CharSequence)this.serverUser) ? InfluxDBFactory.connect(this.serverUrl) : InfluxDBFactory.connect(this.serverUrl, this.serverUser, this.serverPass);
            }
            this.serverUrl = newServerUrl;
            this.serverUser = newServerUser;
            this.serverPass = newServerPass;
        }
        return Optional.ofNullable(this.cachedServer);
    }

    private Writer writer(Series s) {
        return this.writers.computeIfAbsent(s, series -> {
            if (series.getMeasurement().equals("rs_self_loc")) {
                return new Writer(new ThrottledWriter(), SELF_DEDUPE);
            }
            if (series.getMeasurement().equals("rs_activity") || series.getMeasurement().equals("rs_loot")) {
                return new Writer(new AlwaysWriter(), (a, b) -> true);
            }
            if (series.getMeasurement().equals("rs_skilling_items")) {
                return new Writer(new SummingWriter(false), (a, b) -> true);
            }
            return new Writer(new ThrottledWriter(), FULL_DEDUPE);
        });
    }

    public synchronized void flush() {
        Optional<InfluxDB> influx = this.getInflux();
        if (!influx.isPresent()) {
            return;
        }
        BatchPoints.Builder batch = BatchPoints.database(this.config.getDatabase()).retentionPolicy(this.config.getServerRetentionPolicy()).consistency(InfluxDB.ConsistencyLevel.ONE);
        this.writers.forEach((k, v) -> v.flush(batch));
        influx.ifPresent(influxDB -> {
            BatchPoints built = batch.build();
            if (!built.getPoints().isEmpty()) {
                influxDB.write(built);
                if (log.isDebugEnabled()) {
                    log.debug("Writing {}", (Object)built.lineProtocol());
                }
            }
        });
    }

    private static final class SummingWriter
    implements TerminalOp {
        private final boolean floatingPoint;
        private Series series;
        private final Map<String, Number> values = new HashMap<String, Number>();

        public SummingWriter(boolean floatingPoint) {
            this.floatingPoint = floatingPoint;
        }

        @Override
        public Measurement getLastWritten() {
            return null;
        }

        @Override
        public boolean isBlocked() {
            return false;
        }

        @Override
        public synchronized void submit(Measurement m) {
            if (this.series == null) {
                this.series = m.getSeries();
            }
            Preconditions.checkArgument(m.getStringValues().isEmpty(), "Summing writer doesn't support string values");
            for (Map.Entry<String, Number> entry : m.getNumericValues().entrySet()) {
                Number existing = this.values.get(entry.getKey());
                Number output = this.floatingPoint ? (Number)((existing != null ? existing.doubleValue() : 0.0) + entry.getValue().doubleValue()) : (Number)((existing != null ? existing.longValue() : 0L) + entry.getValue().longValue());
                this.values.put(entry.getKey(), output);
            }
        }

        @Override
        public synchronized void flush(BatchPoints.Builder output) {
            if (this.values.isEmpty()) {
                return;
            }
            Measurement.builder().series(this.series).numericValues(this.values).build().toInflux().ifPresent(output::point);
            this.values.clear();
        }
    }

    private static class AlwaysWriter
    implements TerminalOp {
        private final ArrayDeque<Measurement> queued = new ArrayDeque();

        private AlwaysWriter() {
        }

        @Override
        public synchronized Measurement getLastWritten() {
            return this.queued.isEmpty() ? null : this.queued.peekLast();
        }

        @Override
        public boolean isBlocked() {
            return false;
        }

        @Override
        public synchronized void submit(Measurement m) {
            this.queued.add(m);
        }

        @Override
        public synchronized void flush(BatchPoints.Builder output) {
            while (!this.queued.isEmpty()) {
                this.queued.removeFirst().toInflux().ifPresent(output::point);
            }
        }
    }

    private static class ThrottledWriter
    implements TerminalOp {
        private volatile Measurement lastWritten;
        private final AtomicReference<Measurement> waitingForWrite = new AtomicReference();

        private ThrottledWriter() {
        }

        @Override
        public boolean isBlocked() {
            return this.waitingForWrite.get() != null;
        }

        @Override
        public void submit(Measurement m) {
            this.waitingForWrite.set(m);
        }

        @Override
        public void flush(BatchPoints.Builder output) {
            Measurement flush;
            this.lastWritten = flush = (Measurement)this.waitingForWrite.getAndSet(null);
            if (flush != null) {
                flush.toInflux().ifPresent(output::point);
            }
        }

        @Override
        public Measurement getLastWritten() {
            return this.lastWritten;
        }
    }

    private static interface FilterOp {
        public boolean shouldWrite(Measurement var1, Measurement var2);
    }

    private static interface TerminalOp {
        public Measurement getLastWritten();

        public boolean isBlocked();

        public void submit(Measurement var1);

        public void flush(BatchPoints.Builder var1);
    }

    private static class Writer {
        private final TerminalOp terminal;
        private final FilterOp[] filters;

        private Writer(TerminalOp terminal, FilterOp ... filters) {
            this.terminal = terminal;
            this.filters = filters;
        }

        boolean isBlocked() {
            return this.terminal.isBlocked();
        }

        synchronized void submit(Measurement m) {
            Measurement prev = this.terminal.getLastWritten();
            for (FilterOp e : this.filters) {
                if (e.shouldWrite(prev, m)) continue;
                return;
            }
            this.terminal.submit(m);
        }

        synchronized void flush(BatchPoints.Builder output) {
            this.terminal.flush(output);
        }
    }
}

