/*
 * Decompiled with CFR 0.152.
 */
package net.machpi.runelite.influxdb;

import com.google.inject.Provides;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;
import net.machpi.runelite.influxdb.InfluxDbConfig;
import net.machpi.runelite.influxdb.InventoryID2;
import net.machpi.runelite.influxdb.MeasurementCreator;
import net.machpi.runelite.influxdb.SkillingItemTracker;
import net.machpi.runelite.influxdb.activity.ActivityState;
import net.machpi.runelite.influxdb.activity.GameEvent;
import net.machpi.runelite.influxdb.write.InfluxWriter;
import net.runelite.api.Client;
import net.runelite.api.GameState;
import net.runelite.api.Item;
import net.runelite.api.ItemContainer;
import net.runelite.api.Player;
import net.runelite.api.Skill;
import net.runelite.api.WorldType;
import net.runelite.api.coords.LocalPoint;
import net.runelite.api.coords.WorldPoint;
import net.runelite.api.events.GameStateChanged;
import net.runelite.api.events.GameTick;
import net.runelite.api.events.ItemContainerChanged;
import net.runelite.api.events.StatChanged;
import net.runelite.api.events.VarbitChanged;
import net.runelite.api.widgets.Widget;
import net.runelite.api.widgets.WidgetInfo;
import net.runelite.client.config.ConfigManager;
import net.runelite.client.eventbus.Subscribe;
import net.runelite.client.events.ConfigChanged;
import net.runelite.client.plugins.Plugin;
import net.runelite.client.plugins.PluginDescriptor;
import net.runelite.client.plugins.loottracker.LootReceived;
import net.runelite.client.task.Schedule;
import net.runelite.client.util.ExecutorServiceExceptionLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@PluginDescriptor(name="InfluxDB", description="Saves statistics to InfluxDB", tags={"experience", "levels", "stats", "activity", "tracker"})
public class InfluxDbPlugin
extends Plugin {
    private static final Logger log = LoggerFactory.getLogger(InfluxDbPlugin.class);
    private ScheduledFuture<?> flushTask;
    @Inject
    private ConfigManager configManager;
    @Inject
    private InfluxWriter writer;
    @Inject
    private InfluxDbConfig config;
    @Inject
    private Client client;
    @Inject
    private MeasurementCreator measurer;
    @Inject
    private ActivityState activityState;
    @Inject
    private SkillingItemTracker skillingItemTracker;
    private final ScheduledExecutorService executor = new ExecutorServiceExceptionLogger(Executors.newSingleThreadScheduledExecutor());
    private final EnumMap<Skill, Integer> previousStatXp = new EnumMap(Skill.class);
    private GameState prevGameState;
    private boolean varPlayerChanged;
    private String lastMeasuredProfile;
    private int failures = 0;
    private int failureBackoff = 0;

    @Provides
    InfluxDbConfig getConfig(ConfigManager configManager) {
        return (InfluxDbConfig)configManager.getConfig(InfluxDbConfig.class);
    }

    @Subscribe
    public void onStatChanged(StatChanged statChanged) {
        GameEvent gameEvent;
        if (this.measurer.isInLastManStanding()) {
            return;
        }
        if (statChanged.getXp() == 0 || this.client.getGameState() != GameState.LOGGED_IN) {
            return;
        }
        Integer previous = this.previousStatXp.put(statChanged.getSkill(), statChanged.getXp());
        if (previous == null || previous.intValue() == statChanged.getXp()) {
            return;
        }
        this.previousStatXp.put(statChanged.getSkill(), statChanged.getXp());
        if (this.config.writeSkillingItems()) {
            this.skillingItemTracker.onXpGained(statChanged.getSkill(), statChanged.getXp() - previous);
        }
        if (this.config.writeXp()) {
            this.measurer.createXpMeasurement(statChanged.getSkill()).ifPresent(this.writer::submit);
            if (statChanged.getSkill() != Skill.OVERALL) {
                this.measurer.createXpMeasurement(Skill.OVERALL).ifPresent(this.writer::submit);
            }
        }
        if (this.config.writeActivity() && (gameEvent = GameEvent.fromSkill(statChanged.getSkill())) != null) {
            this.activityState.triggerEvent(gameEvent);
        }
    }

    @Subscribe
    public void onGameStateChanged(GameStateChanged event) {
        GameState prev = this.prevGameState;
        this.prevGameState = event.getGameState();
        switch (event.getGameState()) {
            case LOGIN_SCREEN: {
                this.checkForGameStateUpdate();
                break;
            }
            case LOGGING_IN: {
                this.previousStatXp.clear();
                break;
            }
            case LOGGED_IN: {
                if (prev != GameState.LOGGING_IN) break;
                this.checkForGameStateUpdate();
            }
        }
        this.checkForAreaUpdate();
    }

    private void maybeMeasureInitialState() {
        String profile = this.configManager.getRSProfileKey();
        if (profile == null || Objects.equals(profile, this.lastMeasuredProfile)) {
            return;
        }
        this.lastMeasuredProfile = profile;
        if (this.config.writeXp() && !this.measurer.isInLastManStanding()) {
            for (Skill s : Skill.values()) {
                this.measurer.createXpMeasurement(s).ifPresent(this.writer::submit);
            }
        }
        if (this.config.writeKillCount()) {
            String prefix = "killcount." + profile + ".";
            for (String groupAndKey : this.configManager.getConfigurationKeys(prefix)) {
                String boss = groupAndKey.substring(prefix.length());
                this.measurer.createKillCountMeasurement(boss).ifPresent(this.writer::submit);
            }
        }
        if (this.config.writeSelfMeta()) {
            this.measurer.createAchievementMeasurements(this.writer::submit);
        }
        this.checkForGameStateUpdate();
        this.checkForAreaUpdate();
    }

    @Subscribe
    public void onItemContainerChanged(ItemContainerChanged event) {
        ItemContainer container = event.getItemContainer();
        if (container == null) {
            return;
        }
        InventoryID2 id = null;
        for (InventoryID2 val : InventoryID2.values()) {
            if (val.getId() != event.getContainerId()) continue;
            id = val;
            break;
        }
        if (id == InventoryID2.INVENTORY && this.config.writeSkillingItems()) {
            this.skillingItemTracker.onInventoryChanges(container);
        }
        if (this.config.writeBankValue()) {
            if (id != InventoryID2.BANK && id != InventoryID2.SEED_VAULT && id != InventoryID2.COLLECTION_LOG) {
                return;
            }
            if (this.writer.isBlocked(this.measurer.createItemSeries(id, MeasurementCreator.InvValueType.HA))) {
                return;
            }
            Item[] items = container.getItems();
            this.measurer.createItemMeasurements(id, items).forEach(this.writer::submit);
        }
    }

    @Subscribe
    public void onGameTick(GameTick tick) {
        this.maybeMeasureInitialState();
        this.skillingItemTracker.flushIfNeeded();
        if (this.config.writeSelfLoc()) {
            this.writer.submit(this.measurer.createSelfLocMeasurement());
        }
        if (this.config.writeSelfMeta()) {
            this.writer.submit(this.measurer.createSelfMeasurement());
        }
        if (this.varPlayerChanged) {
            if (this.config.writeSelfMeta()) {
                this.measurer.createAchievementMeasurements(this.writer::submit);
            }
            this.varPlayerChanged = false;
        }
    }

    @Subscribe
    public void onConfigChanged(ConfigChanged changed) {
        if ("influxdb".equals(changed.getGroup())) {
            this.failureBackoff = 0;
            if ("writeInterval".equals(changed.getKey())) {
                this.rescheduleFlush();
            }
        }
        this.observeKillCountConfig(changed.getGroup(), changed.getKey());
    }

    private void observeKillCountConfig(String group, String key) {
        if (!this.config.writeKillCount()) {
            return;
        }
        if (!group.equals("killcount") && !group.equals("personalbest")) {
            return;
        }
        this.measurer.createKillCountMeasurement(key).ifPresent(this.writer::submit);
    }

    public void flush() {
        if (this.failureBackoff > 0) {
            --this.failureBackoff;
            return;
        }
        try {
            this.writer.flush();
            this.failures = 0;
        }
        catch (RuntimeException ex) {
            ++this.failures;
            log.error("Failed to write to influxDB " + this.failures + " times", (Throwable)ex);
            this.failureBackoff = Math.min(32, this.failures * this.failures);
        }
    }

    private synchronized void rescheduleFlush() {
        this.unscheduleFlush();
        this.flushTask = this.executor.scheduleWithFixedDelay(this::flush, this.config.writeIntervalSeconds(), this.config.writeIntervalSeconds(), TimeUnit.SECONDS);
    }

    private synchronized void unscheduleFlush() {
        if (this.flushTask != null) {
            this.flushTask.cancel(false);
            this.flushTask = null;
        }
    }

    @Subscribe
    public void onLootReceived(LootReceived event) {
        if (this.config.writeLoot()) {
            this.measurer.createLootMeasurement(event).ifPresent(this.writer::submit);
        }
    }

    @Subscribe
    public void onVarbitChanged(VarbitChanged event) {
        this.varPlayerChanged = true;
        GameEvent gameEvent = GameEvent.fromVarbit(this.client);
        if (gameEvent != null) {
            this.activityState.triggerEvent(gameEvent);
        }
    }

    private void checkForAreaUpdate() {
        if (this.client.getLocalPlayer() == null) {
            return;
        }
        Player localPlayer = this.client.getLocalPlayer();
        int regionId = WorldPoint.fromLocalInstance((Client)this.client, (LocalPoint)localPlayer.getLocalLocation()).getRegionID();
        if (regionId == 0) {
            return;
        }
        EnumSet worldType = this.client.getWorldType();
        GameEvent gameEvent = GameEvent.fromRegion(regionId);
        Widget wildyWidget = this.client.getWidget(WidgetInfo.PVP_WILDERNESS_LEVEL);
        if (GameEvent.MG_NIGHTMARE_ZONE == gameEvent && localPlayer.getWorldLocation().getPlane() == 0) {
            gameEvent = GameEvent.BOSS_KING_BLACK_DRAGON;
        } else if (wildyWidget != null && !wildyWidget.isHidden() && !"".equals(wildyWidget.getText())) {
            gameEvent = GameEvent.WILDERNESS;
        } else if (worldType.contains(WorldType.DEADMAN)) {
            gameEvent = GameEvent.PLAYING_DEADMAN;
        } else if (WorldType.isPvpWorld((Collection)worldType)) {
            gameEvent = GameEvent.PLAYING_PVP;
        } else if (gameEvent == null) {
            gameEvent = GameEvent.IN_GAME;
        }
        this.activityState.triggerEvent(gameEvent);
    }

    private void checkForGameStateUpdate() {
        this.activityState.reset();
        this.activityState.triggerEvent(this.client.getGameState() == GameState.LOGGED_IN ? GameEvent.IN_GAME : GameEvent.IN_MENU);
    }

    @Schedule(period=50L, unit=ChronoUnit.SECONDS)
    public void updateActivity() {
        this.activityState.checkForTimeout();
        this.activityState.measure().ifPresent(this.writer::submit);
    }

    protected void startUp() {
        this.rescheduleFlush();
    }

    protected void shutDown() {
        this.updateActivity();
        this.flush();
        this.unscheduleFlush();
    }
}

