/*
 * Decompiled with CFR 0.152.
 */
package rs117.hd.utils;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import java.io.File;
import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import rs117.hd.utils.ResourcePath;

public class FileWatcher {
    private static final Logger log = LoggerFactory.getLogger(FileWatcher.class);
    private static final WatchEvent.Kind<?>[] eventKinds = new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY};
    private static Thread watcherThread;
    private static Thread runnerThread;
    private static WatchService watchService;
    private static final HashMap<WatchKey, Path> watchKeys;
    private static final ListMultimap<String, Consumer<ResourcePath>> changeHandlers;
    private static final DelayQueue<PendingChange> pendingChanges;

    private static void initialize() throws IOException {
        watchService = FileSystems.getDefault().newWatchService();
        watcherThread = new Thread(() -> {
            try {
                WatchKey watchKey;
                while ((watchKey = watchService.take()) != null) {
                    Path dir = watchKeys.get(watchKey);
                    if (dir == null) {
                        log.error("Unknown WatchKey: " + watchKey);
                        continue;
                    }
                    for (WatchEvent<?> event : watchKey.pollEvents()) {
                        Path path;
                        if (event.kind() == StandardWatchEventKinds.OVERFLOW || (path = dir.resolve((Path)event.context())).toString().endsWith("~")) continue;
                        log.trace("WatchEvent of kind {} for path {}", event.kind(), (Object)path);
                        try {
                            if (event.kind() == StandardWatchEventKinds.ENTRY_CREATE && path.toFile().isDirectory()) {
                                FileWatcher.watchRecursively(path);
                            }
                            Object key = path.toString();
                            ResourcePath resourcePath = ResourcePath.path(new String[]{key});
                            if (path.toFile().isDirectory()) {
                                key = (String)key + File.separator;
                            }
                            for (Map.Entry entry : changeHandlers.entries()) {
                                if (!((String)key).startsWith((String)entry.getKey())) continue;
                                FileWatcher.queuePendingChange(resourcePath, (Consumer)entry.getValue());
                            }
                        }
                        catch (Exception ex) {
                            log.error("Error while handling file change event:", (Throwable)ex);
                        }
                    }
                    watchKey.reset();
                }
            }
            catch (ClosedWatchServiceException watchKey) {
            }
            catch (InterruptedException ex) {
                log.error("Watcher thread interrupted", (Throwable)ex);
            }
        }, FileWatcher.class.getSimpleName() + " Watcher");
        watcherThread.setDaemon(true);
        watcherThread.start();
        runnerThread = new Thread(() -> {
            try {
                PendingChange pending;
                while ((pending = (PendingChange)pendingChanges.poll(100L, TimeUnit.DAYS)) != null) {
                    try {
                        pending.handler.accept(pending.path);
                    }
                    catch (Throwable throwable) {
                        log.error("Error in change handler for path: {}", (Object)pending.path, (Object)throwable);
                    }
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }, FileWatcher.class.getSimpleName() + " Runner");
        runnerThread.setDaemon(true);
        runnerThread.start();
    }

    private static void queuePendingChange(ResourcePath path, Consumer<ResourcePath> handler) {
        PendingChange pendingChange = new PendingChange(path, handler, System.currentTimeMillis() + 200L);
        boolean ignored = pendingChanges.remove(pendingChange);
        pendingChanges.add(pendingChange);
    }

    public static void destroy() {
        if (watchService == null) {
            return;
        }
        try {
            log.debug("Shutting down {}", (Object)FileWatcher.class.getSimpleName());
            changeHandlers.clear();
            watchKeys.clear();
            watchService.close();
            watchService = null;
            if (watcherThread.isAlive()) {
                watcherThread.join();
            }
            runnerThread.interrupt();
            if (runnerThread.isAlive()) {
                runnerThread.join();
            }
        }
        catch (IOException | InterruptedException ex) {
            throw new RuntimeException("Error while closing " + FileWatcher.class.getSimpleName(), ex);
        }
    }

    public static UnregisterCallback watchPath(@NonNull ResourcePath resourcePath, @NonNull Consumer<ResourcePath> changeHandler) {
        if (resourcePath == null) {
            throw new NullPointerException("resourcePath is marked non-null but is null");
        }
        if (changeHandler == null) {
            throw new NullPointerException("changeHandler is marked non-null but is null");
        }
        if (!resourcePath.isFileSystemResource()) {
            throw new IllegalStateException("Only resources on the file system can be watched: " + resourcePath);
        }
        try {
            Consumer<ResourcePath> handler;
            Object key;
            Path path;
            if (watchService == null) {
                FileWatcher.initialize();
            }
            if ((path = resourcePath.toPath()).toFile().isDirectory()) {
                FileWatcher.watchRecursively(path);
                key = path + File.separator;
                handler = changeHandler;
            } else {
                FileWatcher.watchFile(path);
                key = path.toString();
                handler = changed -> {
                    try {
                        if (Files.isSameFile(changed.toPath(), resourcePath.toPath())) {
                            changeHandler.accept((ResourcePath)changed);
                        }
                    }
                    catch (IOException ex) {
                        throw new RuntimeException(ex);
                    }
                };
            }
            changeHandlers.put(key, handler);
            return () -> FileWatcher.lambda$watchPath$3((String)key, handler);
        }
        catch (IOException ex) {
            throw new RuntimeException("Failed to initialize " + FileWatcher.class.getSimpleName(), ex);
        }
    }

    private static void watchFile(Path path) {
        Path dir = path.getParent();
        try {
            watchKeys.put(dir.register(watchService, eventKinds), dir);
            log.debug("Watching {}", (Object)path);
        }
        catch (IOException ex) {
            throw new RuntimeException("Failed to register file watcher for path: " + path, ex);
        }
    }

    private static void watchRecursively(Path path) {
        try {
            log.debug("Watching {}", (Object)path);
            Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    WatchKey key = dir.register(watchService, eventKinds);
                    watchKeys.put(key, dir);
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException ex) {
            throw new RuntimeException("Failed to register recursive file watcher for path: " + path, ex);
        }
    }

    private static /* synthetic */ void lambda$watchPath$3(String key, Consumer handler) {
        changeHandlers.remove((Object)key, (Object)handler);
    }

    static {
        watchKeys = new HashMap();
        changeHandlers = ArrayListMultimap.create();
        pendingChanges = new DelayQueue();
    }

    private static class PendingChange
    implements Delayed {
        final ResourcePath path;
        final Consumer<ResourcePath> handler;
        long delayUntilMillis;

        public boolean equals(Object obj) {
            return obj instanceof PendingChange && this.path.equals(((PendingChange)obj).path) && this.handler.equals(((PendingChange)obj).handler);
        }

        @Override
        public long getDelay(TimeUnit timeUnit) {
            return timeUnit.convert(this.delayUntilMillis - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed delayed) {
            return (int)(this.getDelay(TimeUnit.MILLISECONDS) - delayed.getDelay(TimeUnit.MILLISECONDS));
        }

        public PendingChange(ResourcePath path, Consumer<ResourcePath> handler, long delayUntilMillis) {
            this.path = path;
            this.handler = handler;
            this.delayUntilMillis = delayUntilMillis;
        }
    }

    @FunctionalInterface
    public static interface UnregisterCallback {
        public void unregister();
    }
}

