/*
 * Decompiled with CFR 0.152.
 */
package com.toofifty.xpmeter;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.annotations.Expose;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.toofifty.xpmeter.TrackingMode;
import com.toofifty.xpmeter.Util;
import com.toofifty.xpmeter.XPMeterConfig;
import java.awt.Point;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import net.runelite.api.Skill;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class XPTracker {
    private static final Logger log = LoggerFactory.getLogger(XPTracker.class);
    private static final int ONE_MINUTE = 100;
    private static final int ONE_HOUR = 6000;
    @Inject
    private XPMeterConfig config;
    @Inject
    private Gson gson;
    @Expose
    private final Map<Skill, List<XPGain>> xpGained = new HashMap<Skill, List<XPGain>>();
    @Expose
    private final Map<Skill, Integer> lastXp = new HashMap<Skill, Integer>();
    @Expose
    private final Map<Skill, Integer> startTicks = new HashMap<Skill, Integer>();
    @Expose
    private int currentTick;
    @Expose
    private boolean paused;
    @Expose
    private final Set<Integer> pauses = new HashSet<Integer>();
    @Expose
    private final Set<Integer> logouts = new HashSet<Integer>();
    private Set<Skill> enabledSkills;
    private final Map<Integer, Integer> cache = new HashMap<Integer, Integer>();
    private List<Skill> sortedSkills;
    private int maxXpPerHour;
    private int cacheHits = 0;
    private int cacheMisses = 0;
    private Performance performance;

    public void track(Skill skill, int xp) {
        if (this.paused) {
            return;
        }
        if (this.lastXp.containsKey(skill)) {
            int diff;
            if (!this.xpGained.containsKey(skill)) {
                this.xpGained.put(skill, new ArrayList());
                this.startTicks.put(skill, this.currentTick);
            }
            if ((diff = xp - this.lastXp.get(skill)) != 0) {
                this.xpGained.get(skill).add(new XPGain(this.currentTick, diff));
            }
        }
        this.lastXp.put(skill, xp);
    }

    public boolean isTracking() {
        return !this.lastXp.isEmpty();
    }

    public Set<Skill> getTrackedSkills() {
        return this.xpGained.keySet().stream().filter(skill -> this.enabledSkills.contains(skill)).collect(Collectors.toSet());
    }

    public int getXPPerHourAt(Skill skill, int tick, int windowInterval, TrackingMode trackingMode) {
        int hash = Objects.hash(new Object[]{skill, tick, windowInterval, trackingMode});
        if (!this.cache.containsKey(hash) || this.config.disableCache()) {
            List xpGains = this.xpGained.getOrDefault(skill, List.of());
            int interval = Util.secondsToTicks(windowInterval);
            int elapsed = tick - this.startTicks.getOrDefault(skill, 0);
            int xpGained = xpGains.stream().filter(xpGain -> xpGain.tick <= tick && (trackingMode == TrackingMode.CUMULATIVE || xpGain.tick > tick - interval)).mapToInt(xpGain -> xpGain.xp).sum();
            if (trackingMode == TrackingMode.CUMULATIVE) {
                this.cache.put(hash, xpGained * 6000 / Math.max(100, elapsed));
            } else {
                this.cache.put(hash, xpGained * 6000 / interval);
            }
            ++this.cacheMisses;
        } else {
            ++this.cacheHits;
        }
        return this.cache.get(hash);
    }

    public List<Point> getHistory(Skill skill, int resolution) {
        ArrayList<Point> history = new ArrayList<Point>();
        int startTick = Math.max(this.currentTick - Util.secondsToTicks(this.config.span()), 0);
        int windowInterval = this.config.windowInterval();
        TrackingMode trackingMode = this.config.trackingMode();
        for (int t = startTick; t < this.currentTick; t += Util.secondsToTicks(resolution)) {
            history.add(new Point(t, this.getXPPerHourAt(skill, t, windowInterval, trackingMode)));
        }
        return history;
    }

    public Map<Skill, List<Point>> getAggregate(int maxPixels) {
        this.cacheHits = 0;
        this.cacheMisses = 0;
        Instant start = Instant.now();
        this.maxXpPerHour = 0;
        int pixelResolution = this.config.disableDynamicResolution() ? 1 : this.config.span() / maxPixels;
        int resolution = Math.max(this.config.resolution(), pixelResolution);
        HashMap<Skill, List<Point>> skillHistories = new HashMap<Skill, List<Point>>();
        HashMap<Skill, Integer> skillCurrentRates = new HashMap<Skill, Integer>();
        for (Skill skill : this.getTrackedSkills()) {
            List<Point> history = this.getHistory(skill, resolution);
            skillHistories.put(skill, history);
            Point last = history.get(history.size() - 1);
            skillCurrentRates.put(skill, last == null ? 0 : last.y);
            for (Point point : history) {
                if (point.y <= this.maxXpPerHour) continue;
                this.maxXpPerHour = point.y;
            }
        }
        this.sortedSkills = skillCurrentRates.entrySet().stream().sorted(Comparator.comparingInt(Map.Entry::getValue)).map(Map.Entry::getKey).collect(Collectors.toList());
        long time = Duration.between(start, Instant.now()).toMillis();
        this.performance = new Performance(time, this.cache.size(), this.cacheHits, this.cacheMisses, resolution);
        return skillHistories;
    }

    public void tick() {
        ++this.currentTick;
    }

    public void reset() {
        this.xpGained.clear();
        this.startTicks.clear();
        this.cache.clear();
        this.pauses.clear();
        this.logouts.clear();
        this.currentTick = 0;
    }

    public void clearCache() {
        this.cache.clear();
    }

    public void pause() {
        this.paused = true;
        this.pauses.add(this.currentTick);
    }

    public void unpause() {
        this.paused = false;
    }

    public void trackLogout() {
        this.logouts.add(this.currentTick);
    }

    public String export() {
        return this.gson.newBuilder().excludeFieldsWithoutExposeAnnotation().create().toJson((Object)this);
    }

    public void restore(String json) {
        this.reset();
        JsonObject data = new JsonParser().parse(json).getAsJsonObject();
        for (Map.Entry entry : data.get("xpGained").getAsJsonObject().entrySet()) {
            JsonArray jsonGains = ((JsonElement)entry.getValue()).getAsJsonArray();
            ArrayList<XPGain> gains = new ArrayList<XPGain>();
            for (JsonElement jsonGain : jsonGains) {
                JsonObject gain = jsonGain.getAsJsonObject();
                gains.add(new XPGain(gain.get("tick").getAsInt(), gain.get("xp").getAsInt()));
            }
            this.xpGained.put(Skill.valueOf((String)((String)entry.getKey())), gains);
        }
        for (Map.Entry entry : data.get("lastXp").getAsJsonObject().entrySet()) {
            this.lastXp.put(Skill.valueOf((String)((String)entry.getKey())), ((JsonElement)entry.getValue()).getAsInt());
        }
        for (Map.Entry entry : data.get("startTicks").getAsJsonObject().entrySet()) {
            this.startTicks.put(Skill.valueOf((String)((String)entry.getKey())), ((JsonElement)entry.getValue()).getAsInt());
        }
        this.currentTick = data.get("currentTick").getAsInt();
        this.paused = data.get("paused").getAsBoolean();
        for (JsonElement logout : data.get("pauses").getAsJsonArray()) {
            this.pauses.add(logout.getAsInt());
        }
        for (JsonElement logout : data.get("logouts").getAsJsonArray()) {
            this.logouts.add(logout.getAsInt());
        }
    }

    public int getCurrentTick() {
        return this.currentTick;
    }

    public boolean isPaused() {
        return this.paused;
    }

    public Set<Integer> getPauses() {
        return this.pauses;
    }

    public Set<Integer> getLogouts() {
        return this.logouts;
    }

    public void setEnabledSkills(Set<Skill> enabledSkills) {
        this.enabledSkills = enabledSkills;
    }

    public List<Skill> getSortedSkills() {
        return this.sortedSkills;
    }

    public int getMaxXpPerHour() {
        return this.maxXpPerHour;
    }

    public Performance getPerformance() {
        return this.performance;
    }

    static class Performance {
        private long computeTime;
        private int cacheSize;
        private int cacheHits;
        private int cacheMisses;
        private int renderedResolution;

        public long getComputeTime() {
            return this.computeTime;
        }

        public int getCacheSize() {
            return this.cacheSize;
        }

        public int getCacheHits() {
            return this.cacheHits;
        }

        public int getCacheMisses() {
            return this.cacheMisses;
        }

        public int getRenderedResolution() {
            return this.renderedResolution;
        }

        public Performance(long computeTime, int cacheSize, int cacheHits, int cacheMisses, int renderedResolution) {
            this.computeTime = computeTime;
            this.cacheSize = cacheSize;
            this.cacheHits = cacheHits;
            this.cacheMisses = cacheMisses;
            this.renderedResolution = renderedResolution;
        }
    }

    static class XPGain {
        @Expose
        private int tick;
        @Expose
        private int xp;

        public XPGain(int tick, int xp) {
            this.tick = tick;
            this.xp = xp;
        }
    }
}

