/*
 * Decompiled with CFR 0.152.
 */
package com.flippingutilities.jobs;

import com.flippingutilities.db.TradePersister;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CacheUpdaterJob {
    private static final Logger log = LoggerFactory.getLogger(CacheUpdaterJob.class);
    ScheduledExecutorService executor;
    List<Consumer<String>> subscribers = new ArrayList<Consumer<String>>();
    boolean isBeingShutdownByClient = false;
    Future realTimeUpdateTask;
    Map<String, Long> lastEvents = new HashMap<String, Long>();
    int requiredMinMsSinceLastUpdate = 5;
    int failureCount;
    int failureThreshold = 2;

    public CacheUpdaterJob() {
        this.executor = Executors.newSingleThreadScheduledExecutor();
    }

    public void subscribe(Consumer<String> callback) {
        this.subscribers.add(callback);
    }

    public void start() {
        this.realTimeUpdateTask = this.executor.schedule(this::updateCacheRealTime, 1000L, TimeUnit.MILLISECONDS);
    }

    public void stop() {
        this.isBeingShutdownByClient = true;
        this.realTimeUpdateTask.cancel(true);
    }

    public void updateCacheRealTime() {
        try {
            WatchKey key;
            log.info("starting cache updator job!");
            WatchService watchService = FileSystems.getDefault().newWatchService();
            Path path = TradePersister.PARENT_DIRECTORY.toPath();
            path.register(watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY);
            while ((key = watchService.take()) != null) {
                for (WatchEvent<?> event : key.pollEvents()) {
                    log.info("change in directory for {} with event: {}", event.context(), event.kind());
                    if (!this.isDuplicateEvent(event.context().toString())) {
                        log.info("not duplicate event, firing callbacks");
                        this.subscribers.forEach(subscriber -> subscriber.accept(event.context().toString()));
                        continue;
                    }
                    log.info("duplicate event, not firing callbacks");
                }
                key.reset();
                this.failureCount = 0;
            }
        }
        catch (IOException | InterruptedException e) {
            if (!this.isBeingShutdownByClient) {
                log.info("exception in updateCacheRealTime, Error = {}", (Throwable)e);
                this.onUnexpectedError();
            } else {
                this.onClientShutdown();
            }
        }
        catch (Exception e) {
            log.info("unknown exception in updateCacheRealTime, task is going to stop. Error = {}", (Throwable)e);
        }
    }

    private void onUnexpectedError() {
        log.info("Failure number: {} Error not caused by client shutdown", (Object)this.failureCount);
        ++this.failureCount;
        if (this.failureCount > this.failureThreshold) {
            log.info("number of failures exceeds failure threshold, not scheduling task again");
            return;
        }
        log.info("failure count below threshold, scheduling task again");
        this.realTimeUpdateTask = this.executor.schedule(this::updateCacheRealTime, 1000L, TimeUnit.MILLISECONDS);
    }

    private void onClientShutdown() {
        log.info("shutting down cache updater due to the client shutdown");
    }

    private boolean isDuplicateEvent(String fileName) {
        long lastModified = TradePersister.lastModified(fileName);
        if (this.lastEvents.containsKey(fileName)) {
            long prevModificationTime = this.lastEvents.get(fileName);
            long diffSinceLastModification = Math.abs(lastModified - prevModificationTime);
            if (diffSinceLastModification < (long)this.requiredMinMsSinceLastUpdate) {
                return true;
            }
            this.lastEvents.put(fileName, lastModified);
            return false;
        }
        this.lastEvents.put(fileName, lastModified);
        return false;
    }
}

