/*
 * Decompiled with CFR 0.152.
 */
package com.shootingstartracking;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonSyntaxException;
import com.google.inject.Provides;
import com.shootingstartracking.NotifyType;
import com.shootingstartracking.ShootingStarTrackingConfig;
import com.shootingstartracking.ShootingStarTrackingData;
import com.shootingstartracking.ShootingStarTrackingPanel;
import com.shootingstartracking.WorldHop;
import java.awt.Toolkit;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.inject.Inject;
import javax.swing.SwingUtilities;
import net.runelite.api.ChatMessageType;
import net.runelite.api.Client;
import net.runelite.api.GameState;
import net.runelite.api.events.GameStateChanged;
import net.runelite.api.events.GameTick;
import net.runelite.api.events.WidgetLoaded;
import net.runelite.api.widgets.Widget;
import net.runelite.client.Notifier;
import net.runelite.client.RuneLite;
import net.runelite.client.callback.ClientThread;
import net.runelite.client.chat.ChatColorType;
import net.runelite.client.chat.ChatMessageBuilder;
import net.runelite.client.chat.ChatMessageManager;
import net.runelite.client.chat.QueuedMessage;
import net.runelite.client.config.ConfigManager;
import net.runelite.client.eventbus.Subscribe;
import net.runelite.client.events.ConfigChanged;
import net.runelite.client.plugins.Plugin;
import net.runelite.client.plugins.PluginDescriptor;
import net.runelite.client.task.Schedule;
import net.runelite.client.ui.ClientToolbar;
import net.runelite.client.ui.NavigationButton;
import net.runelite.client.ui.PluginPanel;
import net.runelite.client.util.ImageUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@PluginDescriptor(name="Shooting Star Tracking")
public class ShootingStarTrackingPlugin
extends Plugin {
    private static final Logger log = LoggerFactory.getLogger(ShootingStarTrackingPlugin.class);
    private static final File SAVE_FILE = new File(RuneLite.RUNELITE_DIR, "shooting-star-tracking.json");
    private static final int TELESCOPE_WIDGET_ID = 229;
    private static final ZoneId utcZoneId = ZoneId.of("UTC");
    private static final Pattern minutesThenHoursPattern = Pattern.compile(".* next (\\d+) minutes to (\\d+) hours? (\\d+) .*");
    private static final Pattern minutesPattern = Pattern.compile(".* (\\d+) to (\\d+) .*");
    private static final Pattern hoursPattern = Pattern.compile(".* next (\\d+) hours? (\\d+) minutes? to (\\d+) hours? (\\d+) .*");
    @Inject
    private Client client;
    @Inject
    private ClientThread clientThread;
    @Inject
    private ClientToolbar clientToolbar;
    @Inject
    private ShootingStarTrackingConfig config;
    @Inject
    private ChatMessageManager chatMessageManager;
    @Inject
    private WorldHop worldHop;
    @Inject
    private Notifier notifier;
    private ShootingStarTrackingPanel panel;
    private NavigationButton navButton;
    private boolean displayAsMinutes;
    private List<ShootingStarTrackingData> starData = new ArrayList<ShootingStarTrackingData>();
    private int lastWorld;

    @Subscribe
    public void onWidgetLoaded(WidgetLoaded widgetLoaded) {
        if (widgetLoaded.getGroupId() == 229) {
            this.clientThread.invokeLater(this::extractStarInformation);
        }
    }

    @Subscribe
    public void onConfigChanged(ConfigChanged event) {
        if (!event.getGroup().equals("Shooting Stars Tracking")) {
            return;
        }
        if (event.getKey().equals("displayAsTime")) {
            this.displayAsMinutes = this.config.displayAsMinutes();
            this.panel.updateList(this.starData);
        }
    }

    @Provides
    ShootingStarTrackingConfig getConfig(ConfigManager configManager) {
        return (ShootingStarTrackingConfig)configManager.getConfig(ShootingStarTrackingConfig.class);
    }

    private void extractStarInformation() {
        String widgetText;
        Widget widget = this.client.getWidget(229, 1);
        if (widget == null) {
            return;
        }
        ZonedDateTime minTime = ZonedDateTime.now(utcZoneId);
        try {
            widgetText = widget.getText().replace("<br>", " ");
        }
        catch (NullPointerException e) {
            return;
        }
        Optional<ShootingStarTrackingData.ShootingStarLocations> locationOp = Arrays.stream(ShootingStarTrackingData.ShootingStarLocations.values).filter(o -> widgetText.contains(o.getLocation())).findAny();
        if (!locationOp.isPresent()) {
            log.debug("No match found");
            return;
        }
        int[] minutesInterval = this.minutesInterval(widgetText);
        if (minutesInterval == null) {
            return;
        }
        ZonedDateTime maxTime = minTime.plusMinutes(minutesInterval[1]);
        minTime = minTime.plusMinutes(minutesInterval[0]);
        ShootingStarTrackingData data = new ShootingStarTrackingData(this.client.getWorld(), locationOp.get(), minTime.toInstant().toEpochMilli(), maxTime.toInstant().toEpochMilli());
        this.addToList(data);
        this.save();
        SwingUtilities.invokeLater(() -> this.panel.updateList(this.starData));
    }

    private int[] minutesInterval(String widgetText) {
        Matcher m = minutesThenHoursPattern.matcher(widgetText);
        if (m.find()) {
            int minTime = Integer.parseInt(m.group(1));
            int maxTime = 60 * Integer.parseInt(m.group(2)) + Integer.parseInt(m.group(3));
            return new int[]{minTime, maxTime};
        }
        m = hoursPattern.matcher(widgetText);
        if (m.find()) {
            int minTime = 60 * Integer.parseInt(m.group(1)) + Integer.parseInt(m.group(2));
            int maxTime = 60 * Integer.parseInt(m.group(3)) + Integer.parseInt(m.group(4));
            return new int[]{minTime, maxTime};
        }
        m = minutesPattern.matcher(widgetText);
        if (m.find()) {
            int minTime = Integer.parseInt(m.group(1));
            int maxTime = Integer.parseInt(m.group(2));
            return new int[]{minTime, maxTime};
        }
        return null;
    }

    private void addToList(ShootingStarTrackingData data) {
        ShootingStarTrackingData oldStar = this.starData.stream().filter(star -> data.getWorld() == star.getWorld()).findFirst().orElse(null);
        if (oldStar == null) {
            this.starData.add(data);
            return;
        }
        if (data.getMaxTime() < oldStar.getMinTime()) {
            return;
        }
        this.starData.remove(oldStar);
        if (!data.getLocation().equals((Object)oldStar.getLocation())) {
            this.starData.add(data);
            return;
        }
        long minTime = Math.max(oldStar.getMinTime(), data.getMinTime());
        long maxTime = data.getMaxTime();
        if (data.getMaxTime() >= oldStar.getMinTime()) {
            maxTime = Math.min(oldStar.getMaxTime(), data.getMaxTime());
        }
        ShootingStarTrackingData newStar = new ShootingStarTrackingData(data.getWorld(), data.getLocation(), minTime, maxTime);
        newStar.setNotify(oldStar.isNotify());
        this.starData.add(newStar);
    }

    private void save() {
        String json = new Gson().toJson(this.starData);
        try {
            Files.write(SAVE_FILE.toPath(), json.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Schedule(period=5L, unit=ChronoUnit.SECONDS)
    public void checkDepletedStars() {
        ArrayList<ShootingStarTrackingData> stars = new ArrayList<ShootingStarTrackingData>(this.starData);
        ZonedDateTime now = ZonedDateTime.now(utcZoneId);
        int removeTime = this.config.timeTillRemoveConfig();
        boolean fullUpdate = false;
        for (ShootingStarTrackingData star : this.starData) {
            if (this.checkNotification(star, now)) {
                fullUpdate = true;
            }
            if (star.getMinTime() >= now.minusMinutes(removeTime).toInstant().toEpochMilli()) continue;
            stars.remove(star);
            fullUpdate = true;
        }
        if (fullUpdate) {
            this.starData = stars;
            SwingUtilities.invokeLater(() -> this.panel.updateList(this.starData));
        } else {
            SwingUtilities.invokeLater(() -> this.panel.updateTimes(this.starData));
        }
    }

    private boolean checkNotification(ShootingStarTrackingData star, ZonedDateTime now) {
        if (!star.isNotify()) {
            return false;
        }
        long time = star.getMinTime() + (long)this.config.notifyPercentage() * (star.getMaxTime() - star.getMinTime()) / 100L;
        if (time < now.toInstant().toEpochMilli()) {
            star.setNotify(false);
            String msg = "Star W" + star.getWorld() + " " + star.getLocation().getShortLocation();
            NotifyType type = this.config.notifyType();
            if (type == NotifyType.BOTH || type == NotifyType.NOTIFICATION) {
                this.notifier.notify(msg);
            }
            if (type == NotifyType.BOTH || type == NotifyType.CHAT_MESSAGE) {
                this.sendChatMessage(msg);
            }
            return true;
        }
        return false;
    }

    protected void startUp() throws Exception {
        BufferedImage icon = ImageUtil.loadImageResource(ShootingStarTrackingPlugin.class, (String)"/shooting_star.png");
        this.panel = new ShootingStarTrackingPanel(this);
        this.displayAsMinutes = this.config.displayAsMinutes();
        this.navButton = NavigationButton.builder().tooltip("Shooting Star Tracking").icon(icon).panel((PluginPanel)this.panel).priority(7).build();
        this.clientToolbar.addNavigation(this.navButton);
        try {
            String data = new String(Files.readAllBytes(SAVE_FILE.toPath()));
            this.load(data);
        }
        catch (IOException e) {
            e.printStackTrace();
        }
        this.panel.updateList(this.starData);
    }

    private void load(String data) {
        JsonArray json = (JsonArray)new Gson().fromJson(data, JsonArray.class);
        json.forEach(star -> {
            ShootingStarTrackingData parsedStar = (ShootingStarTrackingData)new Gson().fromJson(star, ShootingStarTrackingData.class);
            this.addToList(parsedStar);
        });
    }

    protected void shutDown() throws Exception {
        this.starData.clear();
        this.clientToolbar.removeNavigation(this.navButton);
    }

    public void removeStar(ShootingStarTrackingData star) {
        this.starData.remove(star);
        this.panel.updateList(this.starData);
        this.save();
    }

    public void importData() {
        try {
            String clipboard = Toolkit.getDefaultToolkit().getSystemClipboard().getData(DataFlavor.stringFlavor).toString().trim();
            this.load(clipboard);
        }
        catch (JsonSyntaxException | UnsupportedFlavorException | IOException er) {
            this.sendChatMessage("Error importing star data.");
            return;
        }
        this.sendChatMessage("Imported star data.");
        this.save();
        this.panel.updateList(this.starData);
    }

    public void sendChatMessage(String chatMessage) {
        if (this.client.getLocalPlayer() == null) {
            return;
        }
        String message = new ChatMessageBuilder().append(ChatColorType.NORMAL).append(chatMessage).build();
        this.chatMessageManager.queue(QueuedMessage.builder().type(ChatMessageType.CONSOLE).runeLiteFormattedMessage(message).build());
    }

    public void exportData() {
        if (this.starData.isEmpty()) {
            this.sendChatMessage("No data to export.");
            return;
        }
        String json = new Gson().toJson(this.starData);
        Toolkit.getDefaultToolkit().getSystemClipboard().setContents(new StringSelection(json), null);
        this.sendChatMessage("Star data exported to clipboard.");
    }

    public void discordFormat() {
        StringBuilder sb = new StringBuilder();
        for (ShootingStarTrackingData star : this.starData) {
            sb.append("`").append(star.getWorld()).append("` - `").append(star.getLocation().getShortLocation()).append("` earliest: ").append("<t:").append(star.getMinTime() / 1000L).append(":R>").append(" latest: ").append("<t:").append(star.getMaxTime() / 1000L).append(":R>").append("\n");
        }
        Toolkit.getDefaultToolkit().getSystemClipboard().setContents(new StringSelection(sb.toString()), null);
        this.sendChatMessage("Star data exported to clipboard in discord format.");
    }

    public void hopTo(ShootingStarTrackingData star) {
        this.worldHop.hop(star.getWorld());
    }

    @Subscribe
    public void onGameTick(GameTick event) {
        this.worldHop.onGameTick();
    }

    public int getWorld() {
        return this.client.getWorld();
    }

    @Subscribe
    public void onGameStateChanged(GameStateChanged event) {
        if (event.getGameState() == GameState.LOGGED_IN && this.client.getWorld() != this.lastWorld) {
            this.lastWorld = this.client.getWorld();
            SwingUtilities.invokeLater(() -> this.panel.updateList(this.starData));
        }
    }

    public void removeWorldsInClipboard() {
        try {
            String clipboard = Toolkit.getDefaultToolkit().getSystemClipboard().getData(DataFlavor.stringFlavor).toString().trim();
            Matcher matcher = Pattern.compile("[wW]([3-5][0-9][0-9])").matcher(clipboard);
            HashSet<Integer> worlds = new HashSet<Integer>();
            while (matcher.find()) {
                try {
                    worlds.add(Integer.parseInt(matcher.group(1)));
                }
                catch (NumberFormatException numberFormatException) {}
            }
            if (worlds.isEmpty()) {
                this.sendChatMessage("No worlds in format w451 found in clipboard.");
                return;
            }
            long timeNow = ZonedDateTime.now(utcZoneId).toInstant().toEpochMilli();
            int sizeBefore = this.starData.size();
            this.starData.removeIf(s -> timeNow >= s.getMinTime() && worlds.contains(s.getWorld()));
            int sizeAfter = this.starData.size();
            this.sendChatMessage("Removed " + (sizeBefore - sizeAfter) + " worlds.");
        }
        catch (JsonSyntaxException | UnsupportedFlavorException | IOException | NumberFormatException er) {
            this.sendChatMessage("Error encountered.");
            return;
        }
        this.save();
        this.panel.updateList(this.starData);
    }

    public WorldHop getWorldHop() {
        return this.worldHop;
    }

    public boolean isDisplayAsMinutes() {
        return this.displayAsMinutes;
    }

    public List<ShootingStarTrackingData> getStarData() {
        return this.starData;
    }

    public void setStarData(List<ShootingStarTrackingData> starData) {
        this.starData = starData;
    }
}

