/*
 * Decompiled with CFR 0.152.
 */
package com.twitchliveloadout.fights;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.twitchliveloadout.TwitchLiveLoadoutConfig;
import com.twitchliveloadout.TwitchLiveLoadoutPlugin;
import com.twitchliveloadout.fights.Fight;
import com.twitchliveloadout.fights.FightSession;
import com.twitchliveloadout.fights.FightSorter;
import com.twitchliveloadout.fights.FightStatistic;
import com.twitchliveloadout.fights.FightStatisticEntry;
import com.twitchliveloadout.fights.FightStatisticProperty;
import java.time.Instant;
import java.util.ArrayList;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import net.runelite.api.Actor;
import net.runelite.api.Client;
import net.runelite.api.GameState;
import net.runelite.api.HeadIcon;
import net.runelite.api.Hitsplat;
import net.runelite.api.IterableHashTable;
import net.runelite.api.NPC;
import net.runelite.api.Player;
import net.runelite.api.Skill;
import net.runelite.api.events.AnimationChanged;
import net.runelite.api.events.FakeXpDrop;
import net.runelite.api.events.GraphicChanged;
import net.runelite.api.events.HitsplatApplied;
import net.runelite.api.events.InteractingChanged;
import net.runelite.api.events.NpcDespawned;
import net.runelite.api.events.PlayerDespawned;
import net.runelite.api.events.StatChanged;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FightStateManager {
    private static final Logger log = LoggerFactory.getLogger(FightStateManager.class);
    private final ConcurrentHashMap<String, Fight> fights = new ConcurrentHashMap();
    private final TwitchLiveLoadoutPlugin plugin;
    private final TwitchLiveLoadoutConfig config;
    private final Client client;
    private static final int ON_GRAPHIC_CHANGED_DELAY = 200;
    private final ScheduledThreadPoolExecutor scheduledExecutor = new ScheduledThreadPoolExecutor(1);
    public static final String HIDDEN_PLAYER_ACTOR_NAME = "__self__";
    public static final float GAME_TICK_DURATION = 0.6f;
    public static final int DEATH_ANIMATION_ID = 836;
    public static final int MAX_FIGHT_AMOUNT = 10;
    public static final int MAX_FINISHED_FIGHT_SESSION_AMOUNT = 1000;
    public static final int MAX_FIGHT_AMOUNT_IN_MEMORY = 50;
    public static final int MAX_FIGHT_DISTANCE = 15;
    public static final int GRAPHIC_HITSPLAT_EXPIRY_TIME_BASE = 1600;
    public static final int GRAPHIC_HITSPLAT_EXPIRY_TIME_PER_SQUARE = 160;
    public static final int GRAPHIC_SKILL_XP_DROP_EXPIRY_TIME = 250;
    private final ConcurrentHashMap<Skill, Instant> lastSkillUpdates = new ConcurrentHashMap();
    private final ConcurrentHashMap<Skill, Integer> lastSkillXp = new ConcurrentHashMap();
    public static final int GRAPHIC_ANIMATION_EXPIRY_TIME = 250;
    private final ConcurrentHashMap<Integer, Instant> lastAnimationUpdates = new ConcurrentHashMap();
    private static final int MAX_INTERACTING_ACTORS_HISTORY = 3;
    private static final int INTERACTING_ACTOR_EXPIRY_TIME = 3000;
    private static final int DEATH_REGISTER_ACTOR_EXPIRY_TIME = 120000;
    private static final boolean DEATH_REGISTER_MIN_DAMAGE_ENABLED = false;
    private static final float DEATH_REGISTER_MIN_DAMAGE_PERCENTAGE = 0.1f;
    private static final int INCOMING_FIGHT_SESSION_AUTO_EXPIRY_TIME = 60000;
    private final ConcurrentHashMap<Actor, Instant> lastInteractingActors = new ConcurrentHashMap();
    private static final String ACTOR_NAME_KEY = "actorNames";
    private static final String ACTOR_TYPE_KEY = "actorTypes";
    private static final String ACTOR_ID_KEY = "actorIds";
    private static final String ACTOR_COMBAT_LEVEL_KEY = "actorCombatLevels";
    private static final String TOTAL_INTERACTING_TICKS_KEY = "totalInteractingTicks";
    private static final String LAST_INTERACTING_TICKS_KEY = "lastInteractingTicks";
    private static final String TOTAL_DURATIONS_KEY = "totalDurations";
    private static final String LAST_DURATIONS_KEY = "lastDurations";
    private static final String UPDATED_ATS_KEY = "updatedAts";
    private static final String SESSION_COUNTERS_KEY = "sessionCounters";
    private static final String STATISTICS_KEY = "statistics";
    public static final Skill NO_SKILL = null;
    public static final int NO_ANIMATION_ID = -1;
    public static final int SINGLE_ANCIENT_ANIMATION_ID = 1978;
    public static final int MULTI_ANCIENT_ANIMATION_ID = 1979;
    public static final int ENTANGLE_ANIMATION_ID = 1161;

    public FightStateManager(TwitchLiveLoadoutPlugin plugin, TwitchLiveLoadoutConfig config, Client client) {
        this.plugin = plugin;
        this.config = config;
        this.client = client;
    }

    public void shutDown() {
        this.clearScheduledUpdates();
        this.scheduledExecutor.shutdown();
    }

    public void onGraphicChanged(GraphicChanged event) {
        final Actor eventActor = event.getActor();
        String eventActorName = this.getFormattedActorName(eventActor);
        IterableHashTable spotAnims = eventActor.getSpotAnims();
        final boolean isInMultiCombatArea = this.isInMultiCombatArea();
        final boolean otherPlayersPresent = this.otherPlayersPresent(eventActor);
        if (spotAnims == null || eventActorName == null) {
            return;
        }
        final ArrayList graphicIds = new ArrayList();
        spotAnims.forEach(spotAnim -> graphicIds.add(spotAnim.getId()));
        log.debug("Scheduling delayed onGraphicChanged, graphic amount: {}", (Object)graphicIds.size());
        this.scheduledExecutor.schedule(new Runnable(){

            @Override
            public void run() {
                try {
                    FightStateManager.this.onGraphicChangedDelayed(eventActor, graphicIds, isInMultiCombatArea, otherPlayersPresent);
                }
                catch (Exception exception) {
                    log.warn("Could not handle an delayed graphic on changed due to the following error: ", (Throwable)exception);
                }
            }
        }, 200L, TimeUnit.MILLISECONDS);
    }

    public void clearScheduledUpdates() {
        this.scheduledExecutor.getQueue().clear();
    }

    public void onGraphicChangedDelayed(Actor eventActor, ArrayList<Integer> graphicIds, boolean isInMultiCombatArea, boolean otherPlayersPresent) {
        Player localPlayer = this.client.getLocalPlayer();
        boolean isLocalPlayer = eventActor == localPlayer;
        log.debug("Handling delayed onGraphicChanged, graphic amount: {}", (Object)graphicIds.size());
        if (localPlayer == null) {
            return;
        }
        Instant now = Instant.now();
        Instant lastInteractedOn = this.lastInteractingActors.get(eventActor);
        boolean lastInteractedWithExpired = lastInteractedOn == null || lastInteractedOn.plusMillis(3000L).isBefore(now);
        boolean validInteractingWith = !lastInteractedWithExpired;
        int distanceTo = localPlayer.getWorldLocation().distanceTo(eventActor.getWorldLocation());
        if (distanceTo > 15) {
            distanceTo = 15;
        }
        for (FightGraphic graphic : FightGraphic.values()) {
            int fightGraphicId = graphic.getGraphicId();
            boolean interactionRequired = graphic.isInteractionRequired();
            FightStatisticProperty property = graphic.getProperty();
            FightStatisticEntry entry = graphic.getEntry();
            if (!graphicIds.contains(fightGraphicId)) continue;
            log.debug("Detected fight graphic, now validating... Graphic ID: {}", (Object)fightGraphicId);
            log.debug("Required skill time until expiry: {}", lastInteractedOn == null ? "N/A" : Long.valueOf(now.toEpochMilli() - lastInteractedOn.plusMillis(3000L).toEpochMilli()));
            if (!isInMultiCombatArea) {
                interactionRequired = true;
            }
            if (!otherPlayersPresent && isInMultiCombatArea) {
                interactionRequired = false;
            }
            if (!isLocalPlayer) {
                if (interactionRequired && !validInteractingWith) continue;
                boolean validSkillUpdates = this.verifySkillsForFightGraphic(graphic);
                boolean validAnimationUpdates = this.verifyAnimationForFightGraphic(graphic);
                if (!validSkillUpdates || !validAnimationUpdates) continue;
            }
            Fight fight = this.ensureValidFight(eventActor);
            if (property == FightStatisticProperty.MISS_COUNTERS || property == FightStatisticProperty.MISS_DAMAGES) {
                FightStatistic statistic = fight.ensureStatistic(eventActor, entry);
                statistic.registerMiss(0);
                continue;
            }
            if (property == FightStatisticProperty.HIT_COUNTERS) {
                FightStatistic statistic = fight.ensureStatistic(eventActor, entry);
                statistic.registerHit(0);
                continue;
            }
            if (property != FightStatisticProperty.HIT_DAMAGES) continue;
            log.debug("The distance to the enemy for the queue expiry time was: {}", (Object)distanceTo);
            int expiryTimeMs = 1600 + 160 * distanceTo;
            fight.queueStatistic(eventActor, entry, property, expiryTimeMs);
        }
    }

    private boolean verifySkillsForFightGraphic(FightGraphic graphic) {
        Instant now = Instant.now();
        Skill requiredSkill = graphic.getRequiredSkill();
        Skill invalidSkill = graphic.getInvalidSkill();
        if (requiredSkill == Skill.MAGIC && !this.config.fightStatisticsSpellsEnabled()) {
            return false;
        }
        if (requiredSkill != null) {
            Instant requiredSkillUpdate = this.lastSkillUpdates.get(requiredSkill);
            if (requiredSkillUpdate == null) {
                return false;
            }
            Instant requiredSkillExpiryTime = requiredSkillUpdate.plusMillis(250L);
            boolean requiredSkillIsExpired = now.isAfter(requiredSkillExpiryTime);
            log.debug("Required skill time until expiry: {}", (Object)(requiredSkillExpiryTime.toEpochMilli() - now.toEpochMilli()));
            if (requiredSkillIsExpired) {
                return false;
            }
        }
        if (invalidSkill != null) {
            Instant invalidSkillUpdate = this.lastSkillUpdates.get(invalidSkill);
            if (invalidSkillUpdate == null) {
                return true;
            }
            Instant invalidSkillExpiryTime = invalidSkillUpdate.plusMillis(250L);
            boolean invalidSkillIsExpired = now.isAfter(invalidSkillExpiryTime);
            log.debug("Invalid skill time until expiry: {}", (Object)(invalidSkillExpiryTime.toEpochMilli() - now.toEpochMilli()));
            if (!invalidSkillIsExpired) {
                return false;
            }
        }
        return true;
    }

    private boolean verifyAnimationForFightGraphic(FightGraphic graphic) {
        Instant now = Instant.now();
        int requiredAnimationId = graphic.getAnimationId();
        Instant requiredSkillUpdate = this.lastAnimationUpdates.get(requiredAnimationId);
        if (requiredAnimationId < 0) {
            return true;
        }
        if (requiredSkillUpdate == null) {
            return false;
        }
        Instant requiredAnimationExpiryTime = requiredSkillUpdate.plusMillis(250L);
        boolean requiredAnimationIsExpired = now.isAfter(requiredAnimationExpiryTime);
        log.debug("Animation time until expiry: {}", (Object)(requiredAnimationExpiryTime.toEpochMilli() - now.toEpochMilli()));
        return !requiredAnimationIsExpired;
    }

    public void onAnimationChanged(AnimationChanged event) {
        Actor eventActor = event.getActor();
        int animationId = eventActor.getAnimation();
        Player localPlayer = this.client.getLocalPlayer();
        if (eventActor == localPlayer) {
            this.lastAnimationUpdates.put(animationId, Instant.now());
        }
        if (eventActor == localPlayer && animationId == 836) {
            if (!this.hasFight(eventActor)) {
                return;
            }
            Fight fight = this.getFight(eventActor);
            fight.finishSession(eventActor);
            fight.increaseSessionCounter();
        }
    }

    public void onHitsplatApplied(HitsplatApplied event) {
        Actor eventActor = event.getActor();
        Hitsplat hitsplat = event.getHitsplat();
        Player player = this.client.getLocalPlayer();
        HeadIcon headIcon = player.getOverheadIcon();
        int hitsplatType = hitsplat.getHitsplatType();
        boolean isOnSelf = this.isLocalPlayer(eventActor);
        if (hitsplatType == 2 || hitsplatType == 5) {
            this.registerExistingFightHitsplat(eventActor, FightStatisticEntry.POISON, hitsplat);
            return;
        }
        if (hitsplatType == 6) {
            this.registerExistingFightHitsplat(eventActor, FightStatisticEntry.HIT_HEAL, hitsplat);
            return;
        }
        if (hitsplatType == 4) {
            return;
        }
        if (!hitsplat.isMine()) {
            if (this.config.fightStatisticsOthersEnabled()) {
                this.registerExistingFightHitsplat(eventActor, FightStatisticEntry.OTHER, hitsplat);
            }
            return;
        }
        this.registerEnsuredFightHitsplat(eventActor, FightStatisticEntry.TOTAL, hitsplat);
        if (!isOnSelf && this.isPlayer(eventActor) && headIcon == HeadIcon.SMITE) {
            this.registerEnsuredFightHitsplat(eventActor, FightStatisticEntry.SMITE, hitsplat);
        }
    }

    public void onNpcDespawned(NpcDespawned npcDespawned) {
        NPC npc = npcDespawned.getNpc();
        Actor eventActor = npcDespawned.getActor();
        if (!npc.isDead()) {
            return;
        }
        this.onActorDespawned(eventActor);
    }

    public void onPlayerDespawned(PlayerDespawned playerDespawned) {
        Player player = playerDespawned.getPlayer();
        Actor eventActor = playerDespawned.getActor();
        if (player.getHealthRatio() != 0) {
            return;
        }
        this.onActorDespawned(eventActor);
    }

    private void onActorDespawned(Actor eventActor) {
        boolean didEnoughDamage;
        double otherDamage;
        if (!this.hasFight(eventActor)) {
            return;
        }
        Instant now = Instant.now();
        Fight fight = this.getFight(eventActor);
        FightSession session = fight.getSession(eventActor);
        if (session == null) {
            return;
        }
        Instant lastUpdate = session.getLastUpdate(true);
        FightStatistic totalStatistic = session.getStatistic(FightStatisticEntry.TOTAL);
        FightStatistic otherStatistic = session.getStatistic(FightStatisticEntry.OTHER);
        double totalDamage = totalStatistic.getHitDamage();
        double allDamage = totalDamage + (otherDamage = (double)otherStatistic.getHitDamage());
        boolean bl = didEnoughDamage = allDamage > 0.0 && totalDamage / allDamage > (double)0.1f;
        if (lastUpdate == null || lastUpdate.plusMillis(120000L).isBefore(now)) {
            return;
        }
        fight.finishSession(eventActor);
        fight.increaseSessionCounter();
    }

    public void onInteractingChanged(InteractingChanged interactingChanged) {
        Actor source = interactingChanged.getSource();
        Actor target = interactingChanged.getTarget();
        Player localPlayer = this.client.getLocalPlayer();
        if (source != localPlayer) {
            return;
        }
        if (target == null) {
            return;
        }
        log.debug("Adding last interacting target to {}", (Object)target.getName());
        this.lastInteractingActors.put(target, Instant.now());
        if (this.lastInteractingActors.size() > 3) {
            this.rotateOldestInteractingActor();
        }
    }

    private void rotateOldestInteractingActor() {
        Actor oldestActor = null;
        for (Actor interactingActor : this.lastInteractingActors.keySet()) {
            Instant oldestLastUpdate;
            Instant lastUpdate = this.lastInteractingActors.get(interactingActor);
            Instant instant = oldestLastUpdate = oldestActor == null ? null : this.lastInteractingActors.get(oldestActor);
            if (oldestLastUpdate != null && !lastUpdate.isBefore(oldestLastUpdate)) continue;
            oldestActor = interactingActor;
        }
        if (oldestActor == null) {
            return;
        }
        this.lastInteractingActors.remove(oldestActor);
    }

    public void onStatChanged(StatChanged event) {
        Integer lastExperience;
        Skill skill = event.getSkill();
        Integer newExperience = this.client.getSkillExperience(skill);
        if (newExperience.equals(lastExperience = this.lastSkillXp.get(skill))) {
            return;
        }
        this.lastSkillXp.put(skill, newExperience);
        this.registerSkillUpdate(skill);
    }

    public void onFakeXpDrop(FakeXpDrop event) {
        Skill skill = event.getSkill();
        this.registerSkillUpdate(skill);
    }

    public void onGameTick() {
        this.registerIdleGameTick();
        this.registerInteractingGameTick();
    }

    private void registerIdleGameTick() {
        if (!this.config.fightStatisticsAutoIdling()) {
            return;
        }
        boolean isLoggedIn = this.client.getGameState() == GameState.LOGGED_IN;
        CopyOnWriteArrayList<String> actorNames = this.getOtherActorNames();
        for (Fight fight : this.fights.values()) {
            if (!fight.isIdling(actorNames) && isLoggedIn) continue;
            fight.queueIdleTicks(1L);
            for (FightSession session : fight.getOngoingSessions()) {
                session.queueIdleTicks(1L);
            }
        }
    }

    private void registerInteractingGameTick() {
        Player localPlayer = this.client.getLocalPlayer();
        if (localPlayer == null) {
            return;
        }
        Actor interactingActor = localPlayer.getInteracting();
        if (interactingActor == null) {
            return;
        }
        this.lastInteractingActors.put(interactingActor, Instant.now());
        if (!this.hasFight(interactingActor)) {
            return;
        }
        Fight fight = this.getFight(interactingActor);
        if (!fight.hasSession(interactingActor)) {
            return;
        }
        FightSession session = fight.getSession(interactingActor);
        session.addInteractingTicks(1L);
    }

    private void registerExistingFightHitsplat(Actor actor, FightStatisticEntry statisticEntry, Hitsplat hitsplat) {
        Fight fight = this.getFight(actor);
        if (fight == null) {
            return;
        }
        if (!this.config.fightStatisticsUnattackedEnabled() && !fight.hasSession(actor)) {
            return;
        }
        this.registerFightHitsplat(fight, actor, statisticEntry, hitsplat);
    }

    private void registerEnsuredFightHitsplat(Actor actor, FightStatisticEntry statisticEntry, Hitsplat hitsplat) {
        Fight fight = this.ensureValidFight(actor);
        this.registerFightHitsplat(fight, actor, statisticEntry, hitsplat);
    }

    private void registerFightHitsplat(Fight fight, Actor actor, FightStatisticEntry statisticEntry, Hitsplat hitsplat) {
        Instant lastUpdate;
        if (fight == null) {
            return;
        }
        Instant now = Instant.now();
        boolean isOnSelf = this.isLocalPlayer(actor);
        if (isOnSelf && (lastUpdate = fight.getLastUpdate()) != null && lastUpdate.plusMillis(60000L).isBefore(now)) {
            fight.finishSession(actor);
        }
        int amount = hitsplat.getAmount();
        int hitsplatType = hitsplat.getHitsplatType();
        FightStatistic statistic = fight.ensureStatistic(actor, statisticEntry);
        if (hitsplat.isMine()) {
            fight.setLastActor(actor);
        }
        if (hitsplat.isMine()) {
            fight.registerQueuedStatistics(actor, amount);
        }
        switch (hitsplatType) {
            case 4: {
                break;
            }
            case 12: 
            case 13: {
                statistic.registerMiss(amount);
                break;
            }
            case 2: 
            case 5: 
            case 6: 
            case 16: 
            case 17: 
            case 18: 
            case 19: 
            case 20: 
            case 21: 
            case 22: 
            case 23: 
            case 24: 
            case 25: 
            case 43: 
            case 44: 
            case 45: 
            case 46: 
            case 47: {
                statistic.registerHit(amount);
            }
        }
    }

    private void registerSkillUpdate(Skill skill) {
        this.lastSkillUpdates.put(skill, Instant.now());
    }

    public Fight ensureValidFight(Actor actor) {
        long lastUpdateDelta;
        if (!this.hasFight(actor)) {
            this.createFight(actor);
        }
        Fight fight = this.getFight(actor);
        Instant now = Instant.now();
        Instant lastUpdate = fight.getLastUpdate();
        long expiryTime = this.config.fightStatisticsExpiryTime() * 60;
        if (lastUpdate != null && (lastUpdateDelta = now.getEpochSecond() - lastUpdate.getEpochSecond()) > expiryTime) {
            this.deleteFight(fight);
            this.createFight(actor);
        }
        return this.getFight(actor);
    }

    public Fight getFight(Actor actor) {
        String actorName = this.getFormattedActorName(actor);
        if (actorName == null) {
            return null;
        }
        return this.fights.get(actorName);
    }

    public boolean hasFight(Actor actor) {
        String actorName = this.getFormattedActorName(actor);
        if (actorName == null) {
            return false;
        }
        return this.fights.containsKey(actorName);
    }

    public void createFight(Actor actor) {
        String localPlayerName = this.client.getLocalPlayer().getName();
        boolean isLocalPlayer = actor instanceof Player && localPlayerName.equals(actor.getName());
        String actorName = this.getFormattedActorName(actor);
        Fight fight = new Fight(actor, actorName, isLocalPlayer);
        while (this.fights.size() >= 50) {
            this.rotateOldestFight();
        }
        log.debug("Creating new fight for actor {}", (Object)actorName);
        this.fights.put(actorName, fight);
        this.updateCombatPanel();
    }

    public void deleteFight(Fight fight) {
        if (fight == null) {
            return;
        }
        log.debug("Removing a fight for actor {}", (Object)fight.getActorName());
        String actorName = fight.getActorName();
        this.fights.remove(actorName);
        this.updateCombatPanel();
    }

    public void rotateOldestFight() {
        Instant oldestLastUpdate = null;
        Fight oldestFight = null;
        for (Fight fight : this.fights.values()) {
            Instant lastUpdate = fight.getLastUpdate();
            if (oldestLastUpdate != null && lastUpdate != null && !lastUpdate.isBefore(oldestLastUpdate)) continue;
            oldestLastUpdate = lastUpdate;
            oldestFight = fight;
        }
        if (oldestFight == null) {
            return;
        }
        this.deleteFight(oldestFight);
    }

    public void deleteAllFights() {
        this.fights.clear();
        this.updateCombatPanel();
    }

    private void updateCombatPanel() {
        this.plugin.getPluginPanel().getCombatPanel().rebuild();
    }

    public JsonObject getFightStatisticsState() {
        int maxFightAmountInState;
        CopyOnWriteArrayList<Fight> includedFights = new CopyOnWriteArrayList<Fight>();
        CopyOnWriteArrayList<FightStatisticEntry> includedStatisticEntries = new CopyOnWriteArrayList<FightStatisticEntry>();
        JsonObject state = new JsonObject();
        JsonObject statistics = new JsonObject();
        JsonArray actorNames = new JsonArray();
        JsonArray actorTypes = new JsonArray();
        JsonArray actorIds = new JsonArray();
        JsonArray actorCombatLevels = new JsonArray();
        JsonArray totalInteractingTicks = new JsonArray();
        JsonArray lastInteractingTicks = new JsonArray();
        JsonArray totalDurations = new JsonArray();
        JsonArray lastDurations = new JsonArray();
        JsonArray sessionCounters = new JsonArray();
        JsonArray updatedAts = new JsonArray();
        for (Fight fight : this.fights.values()) {
            String actorName = fight.getActorName();
            if (actorName == null || actorName.equals("null")) continue;
            includedFights.add(fight);
        }
        int fightAmount = includedFights.size();
        if (fightAmount > (maxFightAmountInState = this.getMaxFightAmountInState())) {
            fightAmount = maxFightAmountInState;
        }
        includedFights.sort(new FightSorter());
        CopyOnWriteArrayList slicedFights = new CopyOnWriteArrayList(includedFights.subList(0, fightAmount));
        state.add(ACTOR_NAME_KEY, (JsonElement)actorNames);
        state.add(ACTOR_TYPE_KEY, (JsonElement)actorTypes);
        state.add(ACTOR_ID_KEY, (JsonElement)actorIds);
        state.add(ACTOR_COMBAT_LEVEL_KEY, (JsonElement)actorCombatLevels);
        state.add(TOTAL_INTERACTING_TICKS_KEY, (JsonElement)totalInteractingTicks);
        state.add(LAST_INTERACTING_TICKS_KEY, (JsonElement)lastInteractingTicks);
        state.add(TOTAL_DURATIONS_KEY, (JsonElement)totalDurations);
        state.add(LAST_DURATIONS_KEY, (JsonElement)lastDurations);
        state.add(SESSION_COUNTERS_KEY, (JsonElement)sessionCounters);
        state.add(UPDATED_ATS_KEY, (JsonElement)updatedAts);
        state.add(STATISTICS_KEY, (JsonElement)statistics);
        for (FightStatisticEntry fightStatisticEntry : FightStatisticEntry.values()) {
            JsonObject fightStatistic = new JsonObject();
            for (FightStatisticProperty property : FightStatisticProperty.values()) {
                fightStatistic.add(property.getKey(), (JsonElement)new JsonArray());
            }
            statistics.add(fightStatisticEntry.getKey(), (JsonElement)fightStatistic);
        }
        for (Fight fight : slicedFights) {
            FightSession totalSession = fight.calculateTotalSession();
            FightSession fightSession = fight.getLastSession();
            String actorName = fight.getActorName();
            Instant lastUpdate = fight.getLastUpdate(true);
            if (fight.getActorType() == ActorType.LOCAL_PLAYER && !this.config.playerInfoEnabled()) {
                actorName = HIDDEN_PLAYER_ACTOR_NAME;
            }
            actorNames.add(actorName);
            actorTypes.add(fight.getActorType().getKey());
            actorIds.add((Number)fight.getActorId());
            actorCombatLevels.add((Number)fight.getActorCombatLevel());
            totalInteractingTicks.add((Number)totalSession.getInteractingTickCounter());
            lastInteractingTicks.add((Number)fightSession.getInteractingTickCounter());
            totalDurations.add((Number)totalSession.getDurationSeconds());
            lastDurations.add((Number)fightSession.getDurationSeconds());
            sessionCounters.add((Number)fight.getSessionCounter());
            updatedAts.add((Number)(lastUpdate == null ? 0L : lastUpdate.getEpochSecond()));
            for (FightStatisticEntry statisticEntry : FightStatisticEntry.values()) {
                FightStatistic totalStatistic = totalSession.getStatistic(statisticEntry);
                FightStatistic lastStatistic = fightSession.getStatistic(statisticEntry);
                JsonObject statisticState = statistics.getAsJsonObject(statisticEntry.getKey());
                for (FightStatisticProperty property : FightStatisticProperty.values()) {
                    long totalValue = totalStatistic.getValueByProperty(property);
                    long lastValue = lastStatistic.getValueByProperty(property);
                    JsonArray totalAndLastValue = new JsonArray();
                    if ((totalStatistic.isEverUpdated() || lastStatistic.isEverUpdated()) && !includedStatisticEntries.contains((Object)statisticEntry)) {
                        includedStatisticEntries.add(statisticEntry);
                    }
                    totalAndLastValue.add((Number)totalValue);
                    totalAndLastValue.add((Number)lastValue);
                    statisticState.getAsJsonArray(property.getKey()).add((JsonElement)totalAndLastValue);
                }
            }
        }
        for (FightStatisticEntry fightStatisticEntry : FightStatisticEntry.values()) {
            if (includedStatisticEntries.contains((Object)fightStatisticEntry)) continue;
            statistics.remove(fightStatisticEntry.getKey());
        }
        return state;
    }

    private boolean isPlayer(Actor actor) {
        return actor instanceof Player;
    }

    private boolean isLocalPlayer(Actor actor) {
        if (!this.isPlayer(actor)) {
            return false;
        }
        Player player = (Player)actor;
        Player localPlayer = this.client.getLocalPlayer();
        return player == localPlayer;
    }

    public String getFormattedActorName(Actor actor) {
        if (actor == null) {
            return null;
        }
        String actorName = actor.getName();
        if (actorName == null) {
            return null;
        }
        String formattedActorName = actorName.replaceAll("\\<[^>]*>", "");
        return formattedActorName;
    }

    public boolean isInMultiCombatArea() {
        int multiCombatVarBit = this.client.getVarbitValue(4605);
        return multiCombatVarBit == 1;
    }

    public boolean otherPlayersPresent(Actor allowedActor) {
        int allowedPlayerAmount = 1;
        if (allowedActor instanceof Player) {
            ++allowedPlayerAmount;
        }
        return this.client.getPlayers().size() > allowedPlayerAmount;
    }

    public CopyOnWriteArrayList<String> getOtherActorNames() {
        CopyOnWriteArrayList<String> actorNames = new CopyOnWriteArrayList<String>();
        CopyOnWriteArrayList actors = new CopyOnWriteArrayList();
        actors.addAll(this.client.getNpcs());
        actors.addAll(this.client.getPlayers());
        for (Actor actor : actors) {
            String actorName = this.getFormattedActorName(actor);
            actorNames.add(actorName);
        }
        return actorNames;
    }

    public int getMaxFightAmountInState() {
        int maxAmount = this.config.fightStatisticsMaxFightAmount();
        if (maxAmount > 10) {
            maxAmount = 10;
        }
        if (maxAmount < 0) {
            maxAmount = 0;
        }
        return maxAmount;
    }

    public ConcurrentHashMap<String, Fight> getFights() {
        return this.fights;
    }

    public static enum ActorType {
        NPC("npc", "npc"),
        PLAYER("player", "player"),
        GAME_OBJECT("gameObject", "gameObject"),
        LOCAL_PLAYER("localPlayer", "self");

        private final String key;
        private final String name;

        private ActorType(String key, String name) {
            this.key = key;
            this.name = name;
        }

        public String getKey() {
            return this.key;
        }

        public String getName() {
            return this.name;
        }
    }

    public static enum FightGraphic {
        ICE_BARRAGE(369, Skill.MAGIC, NO_SKILL, 1979, false, FightStatisticEntry.FREEZE, FightStatisticProperty.HIT_DAMAGES),
        ICE_BLITZ(367, Skill.MAGIC, NO_SKILL, 1978, true, FightStatisticEntry.FREEZE, FightStatisticProperty.HIT_DAMAGES),
        ICE_BURST(363, Skill.MAGIC, NO_SKILL, 1979, false, FightStatisticEntry.FREEZE, FightStatisticProperty.HIT_DAMAGES),
        ICE_RUSH(361, Skill.MAGIC, NO_SKILL, 1978, true, FightStatisticEntry.FREEZE, FightStatisticProperty.HIT_DAMAGES),
        BLOOD_BARRAGE(377, Skill.MAGIC, NO_SKILL, 1979, false, FightStatisticEntry.BLOOD_HEAL, FightStatisticProperty.HIT_DAMAGES),
        BLOOD_BLITZ(375, Skill.MAGIC, NO_SKILL, 1978, true, FightStatisticEntry.BLOOD_HEAL, FightStatisticProperty.HIT_DAMAGES),
        BLOOD_BURST(376, Skill.MAGIC, NO_SKILL, 1979, false, FightStatisticEntry.BLOOD_HEAL, FightStatisticProperty.HIT_DAMAGES),
        BLOOD_RUSH(373, Skill.MAGIC, NO_SKILL, 1978, true, FightStatisticEntry.BLOOD_HEAL, FightStatisticProperty.HIT_DAMAGES),
        ENTANGLE(179, Skill.MAGIC, NO_SKILL, 1161, true, FightStatisticEntry.ENTANGLE, FightStatisticProperty.HIT_DAMAGES),
        SNARE(180, Skill.MAGIC, NO_SKILL, 1161, true, FightStatisticEntry.ENTANGLE, FightStatisticProperty.HIT_DAMAGES),
        BIND(181, Skill.MAGIC, Skill.HITPOINTS, 1161, true, FightStatisticEntry.ENTANGLE, FightStatisticProperty.HIT_COUNTERS),
        SPLASH(85, Skill.MAGIC, NO_SKILL, -1, true, FightStatisticEntry.SPELL, FightStatisticProperty.MISS_COUNTERS);

        private final int graphicId;
        private final Skill requiredSkill;
        private final Skill invalidSkill;
        private final int animationId;
        private final boolean interactionRequired;
        private final FightStatisticEntry entry;
        private final FightStatisticProperty property;

        private FightGraphic(int graphicId, Skill requiredSkill, Skill invalidSkill, int animationId, boolean interactionRequired, FightStatisticEntry entry, FightStatisticProperty property) {
            this.graphicId = graphicId;
            this.requiredSkill = requiredSkill;
            this.invalidSkill = invalidSkill;
            this.animationId = animationId;
            this.interactionRequired = interactionRequired;
            this.entry = entry;
            this.property = property;
        }

        public int getGraphicId() {
            return this.graphicId;
        }

        public Skill getRequiredSkill() {
            return this.requiredSkill;
        }

        public Skill getInvalidSkill() {
            return this.invalidSkill;
        }

        public int getAnimationId() {
            return this.animationId;
        }

        public boolean isInteractionRequired() {
            return this.interactionRequired;
        }

        public FightStatisticEntry getEntry() {
            return this.entry;
        }

        public FightStatisticProperty getProperty() {
            return this.property;
        }
    }
}

