/*
 * Decompiled with CFR 0.152.
 */
package com.hydratereminder;

import com.google.inject.Provides;
import com.hydratereminder.HydrateReminderCommandArgs;
import com.hydratereminder.HydrateReminderConfig;
import java.awt.image.BufferedImage;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import javax.inject.Inject;
import net.runelite.api.ChatMessageType;
import net.runelite.api.Client;
import net.runelite.api.GameState;
import net.runelite.api.IndexedSprite;
import net.runelite.api.events.CommandExecuted;
import net.runelite.api.events.GameStateChanged;
import net.runelite.api.events.GameTick;
import net.runelite.client.Notifier;
import net.runelite.client.config.ConfigManager;
import net.runelite.client.eventbus.Subscribe;
import net.runelite.client.plugins.Plugin;
import net.runelite.client.plugins.PluginDescriptor;
import net.runelite.client.util.ImageUtil;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@PluginDescriptor(name="Hydrate Reminder", description="Reminds players to take a hydration break on a set interval", tags={"hydrate", "health", "reminder", "hydration", "water", "break", "notification"})
public class HydrateReminderPlugin
extends Plugin {
    private static final Logger log = LoggerFactory.getLogger(HydrateReminderPlugin.class);
    private static final List<String> HYDRATE_BREAK_TEXT_LIST = Collections.unmodifiableList(new ArrayList<String>(){
        {
            this.add("It's time for a quick hydration break");
            this.add("Dehydration causes fatigue so take a hydration break");
            this.add("It's time to drink some good ol' water");
            this.add("Stay healthy by taking a hydration break");
            this.add("Time to glug, glug, glug some water");
            this.add("It is now time to take a hydration break");
            this.add("Time to hydrate");
            this.add("Power up with a hydration break now");
            this.add("Got water? It's time to hydrate");
            this.add("Cheers to this hydration break");
            this.add("Hydration time is now");
            this.add("Fuel up with a hydration break");
        }
    });
    private static final List<String> HYDRATE_WELCOME_TEXT_LIST = Collections.unmodifiableList(new ArrayList<String>(){
        {
            this.add("Don't forget to stay hydrated.");
            this.add("Type \"::hydrate help\" in chat to view available commands.");
            this.add("Stay cool. Stay awesome. Stay hydrated.");
            this.add("Keep calm and stay hydrated.");
            this.add("Cheers to staying hydrated!");
            this.add("Keep the geyser titans happy by staying hydrated.");
            this.add("Hydration is love. Hydration is life.");
            this.add("Out of water? Cast humidify to stay hydrated.");
            this.add("It costs zero water runes to stay hydrated.");
            this.add("Check out the hydrate commands by typing \"::hydrate help\" in chat.");
            this.add("A hydrated adventurer is an unstoppable adventurer.");
            this.add("It's dangerous to go alone. Stay hydrated!");
        }
    });
    private static final String RUNELITE_COMMAND_PREFIX = "::";
    private static final String HYDRATE_COMMAND_NAME = "hydrate";
    @Inject
    private Client client;
    @Inject
    private HydrateReminderConfig config;
    @Inject
    private Notifier notifier;
    private Instant lastHydrateInstant = Instant.now();
    private int hydrateEmojiId = -1;
    private Boolean isFirstGameTick = true;

    @Provides
    protected HydrateReminderConfig provideConfig(ConfigManager configManager) {
        return (HydrateReminderConfig)configManager.getConfig(HydrateReminderConfig.class);
    }

    @Subscribe
    public void onGameStateChanged(GameStateChanged gameStateChanged) {
        if (gameStateChanged.getGameState() == GameState.LOGGING_IN) {
            this.isFirstGameTick = true;
            this.resetHydrateReminderTimeInterval();
            log.debug("Hydrate Reminder plugin interval timer started");
        }
    }

    private void loadHydrateEmoji() {
        IndexedSprite[] modIcons = this.client.getModIcons();
        if (modIcons != null) {
            IndexedSprite[] newModIcons = Arrays.copyOf(modIcons, modIcons.length + 1);
            try {
                IndexedSprite hydrateSprite;
                BufferedImage hydrateIcon = ImageUtil.loadImageResource(((Object)((Object)this)).getClass(), (String)"water_icon.png");
                newModIcons[modIcons.length] = hydrateSprite = ImageUtil.getImageIndexedSprite((BufferedImage)hydrateIcon, (Client)this.client);
            }
            catch (Exception e) {
                log.warn("Failed to load hydrate emoji sprite", (Throwable)e);
            }
            this.hydrateEmojiId = modIcons.length;
            this.client.setModIcons(newModIcons);
            log.debug("Successfully loaded hydrate emoji sprite");
        }
    }

    private void sendHydrateWelcomeChatMessage() {
        Random randomGenerator = new Random();
        String hydrateWelcomeMessage = HYDRATE_WELCOME_TEXT_LIST.get(randomGenerator.nextInt(HYDRATE_WELCOME_TEXT_LIST.size()));
        this.sendHydrateEmojiChatMessage(ChatMessageType.GAMEMESSAGE, hydrateWelcomeMessage);
    }

    @Subscribe
    public void onCommandExecuted(CommandExecuted commandExecuted) {
        Object[] args;
        if (commandExecuted.getCommand().equalsIgnoreCase(HYDRATE_COMMAND_NAME) && ArrayUtils.isNotEmpty((Object[])(args = commandExecuted.getArguments()))) {
            try {
                HydrateReminderCommandArgs arg = HydrateReminderCommandArgs.valueOf(((String)args[0]).toUpperCase());
                switch (arg) {
                    case NEXT: {
                        this.handleHydrateNextCommand();
                        break;
                    }
                    case PREV: {
                        this.handleHydratePrevCommand();
                        break;
                    }
                    case RESET: {
                        this.handleHydrateResetCommand();
                        break;
                    }
                    case HELP: {
                        this.handleHydrateHelpCommand();
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException();
                    }
                }
            }
            catch (IllegalArgumentException e) {
                String invalidArgString = String.format("%s%s %s is not a valid command", RUNELITE_COMMAND_PREFIX, HYDRATE_COMMAND_NAME, args[0]);
                this.sendHydrateEmojiChatMessage(ChatMessageType.GAMEMESSAGE, invalidArgString);
                this.handleHydrateHelpCommand();
            }
        }
    }

    private void handleHydrateNextCommand() {
        Instant nextHydrateReminderInstant = this.getNextHydrateReminderInstant();
        Duration timeUntilNextBreak = Duration.between(Instant.now(), nextHydrateReminderInstant);
        String timeString = this.getTimeDisplay(timeUntilNextBreak);
        String message = timeString + " until the next hydration break.";
        this.sendHydrateEmojiChatMessage(ChatMessageType.GAMEMESSAGE, message);
    }

    private void handleHydratePrevCommand() {
        Duration timeSinceLastBreak = Duration.between(this.lastHydrateInstant, Instant.now());
        String timeString = this.getTimeDisplay(timeSinceLastBreak);
        String message = timeString + " since the last hydration break.";
        this.sendHydrateEmojiChatMessage(ChatMessageType.GAMEMESSAGE, message);
    }

    protected String getTimeDisplay(Duration duration) {
        int hours = Math.toIntExact(duration.toHours());
        int minutes = Math.toIntExact(duration.toMinutes() % 60L);
        int seconds = Math.toIntExact(duration.toMillis() / 1000L % 60L);
        StringBuilder timeDisplayBuilder = new StringBuilder();
        if (hours > 0) {
            timeDisplayBuilder.append(hours != 1 ? hours + " hours " : hours + " hour ");
        }
        if (minutes > 0 || hours > 0) {
            timeDisplayBuilder.append(minutes != 1 ? minutes + " minutes " : minutes + " minute ");
        }
        timeDisplayBuilder.append(seconds != 1 ? seconds + " seconds" : seconds + " second");
        return timeDisplayBuilder.toString();
    }

    private void handleHydrateResetCommand() {
        this.resetHydrateReminderTimeInterval();
        String resetString = "Hydrate reminder interval has been successfully reset.";
        this.sendHydrateEmojiChatMessage(ChatMessageType.GAMEMESSAGE, "Hydrate reminder interval has been successfully reset.");
    }

    private void handleHydrateHelpCommand() {
        StringBuilder commandList = new StringBuilder();
        String listSeparator = ", ";
        for (HydrateReminderCommandArgs arg : HydrateReminderCommandArgs.values()) {
            if (commandList.length() > 0) {
                commandList.append(", ");
            }
            commandList.append(arg.toString());
        }
        String helpString = String.format("Available commands: %s%s %s", RUNELITE_COMMAND_PREFIX, HYDRATE_COMMAND_NAME, commandList);
        this.sendHydrateEmojiChatMessage(ChatMessageType.GAMEMESSAGE, helpString);
    }

    @Subscribe
    public void onGameTick(GameTick event) {
        Instant nextHydrateReminderInstant;
        if (this.isFirstGameTick.booleanValue() && this.config.hydrateReminderWelcomeMessageEnabled()) {
            if (this.hydrateEmojiId == -1) {
                this.loadHydrateEmoji();
            }
            this.sendHydrateWelcomeChatMessage();
            this.isFirstGameTick = false;
        }
        if ((nextHydrateReminderInstant = this.getNextHydrateReminderInstant()).compareTo(Instant.now()) < 0) {
            this.handleHydrateReminderDispatch();
            this.resetHydrateReminderTimeInterval();
        }
    }

    private Instant getNextHydrateReminderInstant() {
        Duration hydrateReminderDuration = Duration.ofMinutes(this.config.hydrateReminderInterval());
        return this.lastHydrateInstant.plus(hydrateReminderDuration);
    }

    private void resetHydrateReminderTimeInterval() {
        this.lastHydrateInstant = Instant.now();
    }

    private void handleHydrateReminderDispatch() {
        String hydrateReminderMessage = this.getHydrateReminderMessage();
        if (this.config.hydrateReminderChatMessageEnabled()) {
            this.sendHydrateReminderChatMessage(hydrateReminderMessage);
        }
        if (this.config.hydrateReminderComputerNotificationEnabled()) {
            this.sendHydrateReminderNotification(hydrateReminderMessage);
        }
    }

    private String getHydrateReminderMessage() {
        Random randomGenerator = new Random();
        String playerName = Objects.requireNonNull(this.client.getLocalPlayer()).getName();
        String hydrateReminderMessage = HYDRATE_BREAK_TEXT_LIST.get(randomGenerator.nextInt(HYDRATE_BREAK_TEXT_LIST.size()));
        return String.format("%s, %s.", hydrateReminderMessage, playerName);
    }

    private void sendHydrateReminderChatMessage(String message) {
        ChatMessageType chatMessageType = this.getChatNotificationMessageType();
        this.sendHydrateEmojiChatMessage(chatMessageType, message);
        log.debug(String.format("Successfully sent chat notification of type: %s", chatMessageType.toString()));
    }

    private void sendHydrateEmojiChatMessage(ChatMessageType type, String message) {
        if (this.hydrateEmojiId == -1) {
            this.client.addChatMessage(type, "", message, null);
            return;
        }
        String hydrateEmoji = String.format("<img=%d>", this.hydrateEmojiId);
        StringBuilder hydrateMessage = new StringBuilder();
        String sender = hydrateEmoji;
        if (type != ChatMessageType.FRIENDSCHAT) {
            hydrateMessage.append(hydrateEmoji);
            hydrateMessage.append(" ");
            sender = null;
        }
        hydrateMessage.append(message);
        this.client.addChatMessage(type, "", hydrateMessage.toString(), sender);
    }

    private void sendHydrateReminderNotification(String message) {
        this.notifier.notify(message);
        log.debug("Successfully sent computer notification");
    }

    private ChatMessageType getChatNotificationMessageType() {
        ChatMessageType chatMessageType;
        switch (this.config.hydrateReminderChatMessageType()) {
            case BROADCASTMESSAGE: {
                chatMessageType = ChatMessageType.BROADCAST;
                break;
            }
            case PUBLICCHAT: {
                chatMessageType = ChatMessageType.PUBLICCHAT;
                break;
            }
            case CLANCHAT: {
                chatMessageType = ChatMessageType.FRIENDSCHAT;
                break;
            }
            default: {
                chatMessageType = ChatMessageType.GAMEMESSAGE;
            }
        }
        return chatMessageType;
    }
}

